#!/bin/bash

# AirwavePBX v1.1.0 Management Script (Go Edition)
# Bootstrap script for installation, updates, and system management

set -e

# Configuration
VERSION="1.1.0"
UPDATE_SERVER="https://versions.airwavepbx.com"
SCRIPT_NAME="airwavepbx.sh"
INSTALL_SCRIPT="install.sh"
BINARY_NAME="airwavepbx"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
BOLD='\033[1m'
NC='\033[0m'

# Paths
AIRWAVE_HOME="/opt/airwavepbx"
BINARY_PATH="${AIRWAVE_HOME}/${BINARY_NAME}"
SERVICE_NAME="airwavepbx.service"
LOG_DIR="/var/log/airwavepbx"
CONFIG_DIR="/etc/airwavepbx"
BACKUP_DIR="/var/backups/airwavepbx"

# Utility functions
print_header() {
    clear
    echo ""
    echo -e "${BOLD}${BLUE}╔══════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${BOLD}${BLUE}║${NC}${BOLD}                    AirwavePBX v${VERSION} Manager                     ${NC}${BOLD}${BLUE}║${NC}"
    echo -e "${BOLD}${BLUE}║${NC}${BOLD}                        Go Edition                               ${NC}${BOLD}${BLUE}║${NC}"
    echo -e "${BOLD}${BLUE}╚══════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
}

print_info() {
    echo -e "${BLUE}ℹ $1${NC}"
}

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_step() {
    echo ""
    echo -e "${BOLD}${CYAN}▶ $1${NC}"
    echo "────────────────────────────────────────────────────────────────"
}

# Check if running as root
check_root() {
    if [[ $EUID -ne 0 ]]; then
        print_error "This script must be run as root (use sudo)"
        exit 1
    fi
}

# Get system status
get_system_status() {
    # Check if binary exists
    if [ -f "$BINARY_PATH" ]; then
        AIRWAVE_INSTALLED=true
    else
        AIRWAVE_INSTALLED=false
    fi
    
    # Check service status
    if systemctl is-active --quiet "$SERVICE_NAME" 2>/dev/null; then
        AIRWAVE_RUNNING=true
        AIRWAVE_STATUS="running"
    elif systemctl is-enabled --quiet "$SERVICE_NAME" 2>/dev/null; then
        AIRWAVE_RUNNING=false
        AIRWAVE_STATUS="stopped"
    else
        AIRWAVE_RUNNING=false
        AIRWAVE_STATUS="not installed"
    fi
    
    # Check Asterisk status
    if systemctl is-active --quiet asterisk 2>/dev/null; then
        ASTERISK_RUNNING=true
        ASTERISK_STATUS="running"
    elif systemctl is-enabled --quiet asterisk 2>/dev/null; then
        ASTERISK_RUNNING=false
        ASTERISK_STATUS="stopped"
    else
        ASTERISK_RUNNING=false
        ASTERISK_STATUS="not installed"
    fi
}

# Display system status
show_status() {
    get_system_status
    
    echo -e "${BOLD}${CYAN}System Status:${NC}"
    echo "──────────────────────────────────────────────────────────────"
    
    # AirwavePBX Status
    if [ "$AIRWAVE_INSTALLED" = true ]; then
        echo -e "AirwavePBX: ${GREEN}Installed${NC}"
        if [ "$AIRWAVE_RUNNING" = true ]; then
            echo -e "Service:    ${GREEN}● Running${NC}"
        else
            echo -e "Service:    ${RED}● Stopped${NC}"
        fi
        
        # Get version from binary
        if [ -x "$BINARY_PATH" ]; then
            BINARY_VERSION=$("$BINARY_PATH" -version 2>/dev/null | head -n1 | awk '{print $2}' || echo "Unknown")
            echo -e "Version:    ${CYAN}${BINARY_VERSION}${NC}"
        fi
    else
        echo -e "AirwavePBX: ${RED}Not Installed${NC}"
    fi
    
    echo ""
    
    # Asterisk Status
    if [ "$ASTERISK_STATUS" != "not installed" ]; then
        echo -e "Asterisk:   ${GREEN}Installed${NC}"
        if [ "$ASTERISK_RUNNING" = true ]; then
            echo -e "Service:    ${GREEN}● Running${NC}"
        else
            echo -e "Service:    ${RED}● Stopped${NC}"
        fi
        
        # Get Asterisk version
        ASTERISK_VERSION=$(asterisk -V 2>/dev/null | head -n1 || echo "Unknown")
        echo -e "Version:    ${CYAN}${ASTERISK_VERSION}${NC}"
    else
        echo -e "Asterisk:   ${RED}Not Installed${NC}"
    fi
    
    echo ""
}

# Check for updates
check_for_updates() {
    print_step "Checking for Updates"
    
    print_info "Checking ${UPDATE_SERVER} for latest version..."
    
    # Try to fetch latest version from update server
    LATEST_VERSION=$(curl -s --connect-timeout 10 "${UPDATE_SERVER}/latest-version.txt" 2>/dev/null || echo "")
    
    if [ -z "$LATEST_VERSION" ]; then
        print_warning "Unable to connect to update server"
        return 1
    fi
    
    print_info "Current version: v${VERSION}"
    print_info "Latest version:  v${LATEST_VERSION}"
    
    if [ "$VERSION" = "$LATEST_VERSION" ]; then
        print_success "System is up to date"
        return 0
    else
        print_warning "Update available: v${VERSION} → v${LATEST_VERSION}"
        return 2
    fi
}

# Download and install update
install_update() {
    print_step "Installing Update"
    
    check_for_updates
    if [ $? -eq 0 ]; then
        print_info "No updates available"
        return 0
    fi
    
    print_info "Downloading AirwavePBX v${LATEST_VERSION}..."
    
    # Create temporary directory
    UPDATE_DIR="/tmp/airwave-update-$(date +%s)"
    mkdir -p "$UPDATE_DIR"
    cd "$UPDATE_DIR"
    
    # Download package
    PACKAGE_NAME="airwavepbx-${LATEST_VERSION}-go-linux-amd64.tar.gz"
    if ! wget -q "${UPDATE_SERVER}/${PACKAGE_NAME}"; then
        print_error "Failed to download update package"
        rm -rf "$UPDATE_DIR"
        return 1
    fi
    
    # Download checksums
    if ! wget -q "${UPDATE_SERVER}/${PACKAGE_NAME}.sha256"; then
        print_error "Failed to download checksums"
        rm -rf "$UPDATE_DIR"
        return 1
    fi
    
    # Verify checksum
    print_info "Verifying package integrity..."
    if ! sha256sum -c "${PACKAGE_NAME}.sha256" >/dev/null 2>&1; then
        print_error "Package verification failed"
        rm -rf "$UPDATE_DIR"
        return 1
    fi
    
    # Extract package
    print_info "Extracting update package..."
    tar -xzf "$PACKAGE_NAME"
    
    # Run installer in update mode
    cd "airwavepbx-${LATEST_VERSION}-go-linux-amd64"
    print_info "Running installer in update mode..."
    ./install.sh --update
    
    # Cleanup
    cd /
    rm -rf "$UPDATE_DIR"
    
    print_success "Update completed successfully"
}

# Service management functions
start_services() {
    print_step "Starting Services"
    
    print_info "Starting Asterisk..."
    systemctl start asterisk
    
    print_info "Starting AirwavePBX..."
    systemctl start "$SERVICE_NAME"
    
    sleep 3
    
    if systemctl is-active --quiet asterisk && systemctl is-active --quiet "$SERVICE_NAME"; then
        print_success "All services started successfully"
    else
        print_warning "Some services may have startup issues"
    fi
}

stop_services() {
    print_step "Stopping Services"
    
    print_info "Stopping AirwavePBX..."
    systemctl stop "$SERVICE_NAME" 2>/dev/null || true
    
    print_info "Stopping Asterisk..."
    systemctl stop asterisk 2>/dev/null || true
    
    print_success "Services stopped"
}

restart_services() {
    print_step "Restarting Services"
    
    stop_services
    sleep 2
    start_services
}

# Show service logs
show_logs() {
    echo "Select logs to view:"
    echo "1) AirwavePBX logs"
    echo "2) Asterisk logs"
    echo "3) System logs (journalctl)"
    echo "4) Installation logs"
    echo ""
    read -p "Choice [1-4]: " log_choice
    
    case $log_choice in
        1)
            print_info "Showing AirwavePBX logs..."
            journalctl -u "$SERVICE_NAME" -f --no-pager
            ;;
        2)
            print_info "Showing Asterisk logs..."
            if [ -f "/var/log/asterisk/messages" ]; then
                tail -f /var/log/asterisk/messages
            else
                journalctl -u asterisk -f --no-pager
            fi
            ;;
        3)
            print_info "Showing system logs..."
            journalctl -f --no-pager
            ;;
        4)
            print_info "Showing installation logs..."
            if [ -f "${LOG_DIR}/install.log" ]; then
                tail -f "${LOG_DIR}/install.log"
            else
                print_warning "Installation log not found"
            fi
            ;;
        *)
            print_error "Invalid choice"
            ;;
    esac
}

# Backup system
create_backup() {
    print_step "Creating System Backup"
    
    BACKUP_NAME="airwavepbx-backup-$(date +%Y%m%d-%H%M%S)"
    BACKUP_PATH="${BACKUP_DIR}/${BACKUP_NAME}"
    
    mkdir -p "$BACKUP_PATH"
    
    print_info "Creating backup in ${BACKUP_PATH}..."
    
    # Backup configuration
    if [ -d "$CONFIG_DIR" ]; then
        cp -r "$CONFIG_DIR" "${BACKUP_PATH}/config"
    fi
    
    # Backup data directory
    if [ -d "${AIRWAVE_HOME}/data" ]; then
        cp -r "${AIRWAVE_HOME}/data" "${BACKUP_PATH}/data"
    fi
    
    # Backup Asterisk config
    if [ -d "/etc/asterisk" ]; then
        cp -r "/etc/asterisk" "${BACKUP_PATH}/asterisk"
    fi
    
    # Create tarball
    cd "$BACKUP_DIR"
    tar -czf "${BACKUP_NAME}.tar.gz" "$BACKUP_NAME"
    rm -rf "$BACKUP_NAME"
    
    print_success "Backup created: ${BACKUP_DIR}/${BACKUP_NAME}.tar.gz"
}

# Install AirwavePBX
install_airwavepbx() {
    print_step "Installing AirwavePBX v1.1.0"
    
    # Create temporary directory
    INSTALL_DIR="/tmp/airwave-install-$(date +%s)"
    mkdir -p "$INSTALL_DIR"
    cd "$INSTALL_DIR"
    
    print_info "Downloading AirwavePBX v${VERSION}..."
    PACKAGE_NAME="airwavepbx-${VERSION}-go-linux-amd64.tar.gz"
    
    if ! wget -q "${UPDATE_SERVER}/${PACKAGE_NAME}"; then
        print_error "Failed to download installation package"
        rm -rf "$INSTALL_DIR"
        return 1
    fi
    
    # Download and verify checksum
    if wget -q "${UPDATE_SERVER}/${PACKAGE_NAME}.sha256"; then
        print_info "Verifying package integrity..."
        if ! sha256sum -c "${PACKAGE_NAME}.sha256" >/dev/null 2>&1; then
            print_error "Package verification failed"
            rm -rf "$INSTALL_DIR"
            return 1
        fi
        print_success "Package verified successfully"
    fi
    
    # Extract package
    print_info "Extracting installation package..."
    tar -xzf "$PACKAGE_NAME"
    
    # Run installer
    cd "airwavepbx-${VERSION}-go-linux-amd64"
    print_info "Running installer..."
    ./install.sh
    
    # Cleanup
    cd /
    rm -rf "$INSTALL_DIR"
    
    print_success "Installation completed"
}

# Main menu
show_main_menu() {
    while true; do
        print_header
        show_status
        
        echo -e "${BOLD}${CYAN}Management Options:${NC}"
        echo "──────────────────────────────────────────────────────────────"
        
        if [ "$AIRWAVE_INSTALLED" = false ]; then
            echo "1) Install AirwavePBX"
            echo "2) Exit"
            echo ""
            read -p "Select option [1-2]: " choice
            
            case $choice in
                1) install_airwavepbx; read -p "Press Enter to continue..." ;;
                2) echo "Goodbye!"; exit 0 ;;
                *) print_error "Invalid option"; sleep 2 ;;
            esac
        else
            echo "1) Start Services"
            echo "2) Stop Services"
            echo "3) Restart Services"
            echo "4) View Logs"
            echo "5) Check for Updates"
            echo "6) Install Update"
            echo "7) Create Backup"
            echo "8) System Diagnostics"
            echo "9) Exit"
            echo ""
            read -p "Select option [1-9]: " choice
            
            case $choice in
                1) start_services; read -p "Press Enter to continue..." ;;
                2) stop_services; read -p "Press Enter to continue..." ;;
                3) restart_services; read -p "Press Enter to continue..." ;;
                4) show_logs ;;
                5) check_for_updates; read -p "Press Enter to continue..." ;;
                6) install_update; read -p "Press Enter to continue..." ;;
                7) create_backup; read -p "Press Enter to continue..." ;;
                8) run_diagnostics; read -p "Press Enter to continue..." ;;
                9) echo "Goodbye!"; exit 0 ;;
                *) print_error "Invalid option"; sleep 2 ;;
            esac
        fi
    done
}

# Simple diagnostics
run_diagnostics() {
    print_step "System Diagnostics"
    
    echo "🔍 Checking system health..."
    echo ""
    
    # Check disk space
    echo "💾 Disk Usage:"
    df -h / | tail -n1
    echo ""
    
    # Check memory
    echo "🧠 Memory Usage:"
    free -h | grep Mem
    echo ""
    
    # Check services
    echo "⚙️  Service Status:"
    systemctl is-active asterisk && echo "  ✓ Asterisk: Running" || echo "  ✗ Asterisk: Stopped"
    systemctl is-active "$SERVICE_NAME" && echo "  ✓ AirwavePBX: Running" || echo "  ✗ AirwavePBX: Stopped"
    echo ""
    
    # Check network
    echo "🌐 Network Connectivity:"
    if ping -c 1 8.8.8.8 >/dev/null 2>&1; then
        echo "  ✓ Internet connectivity: OK"
    else
        echo "  ✗ Internet connectivity: Failed"
    fi
    
    # Check ports
    echo ""
    echo "🔌 Port Status:"
    netstat -tuln | grep ":5060 " >/dev/null && echo "  ✓ SIP (5060): Listening" || echo "  ✗ SIP (5060): Not listening"
    netstat -tuln | grep ":8443 " >/dev/null && echo "  ✓ HTTPS (8443): Listening" || echo "  ✗ HTTPS (8443): Not listening"
}

# Handle command line arguments
if [ $# -gt 0 ]; then
    case "$1" in
        --install)
            check_root
            install_airwavepbx
            exit 0
            ;;
        --update)
            check_root
            install_update
            exit 0
            ;;
        --status)
            get_system_status
            show_status
            exit 0
            ;;
        --start)
            check_root
            start_services
            exit 0
            ;;
        --stop)
            check_root
            stop_services
            exit 0
            ;;
        --restart)
            check_root
            restart_services
            exit 0
            ;;
        --backup)
            check_root
            create_backup
            exit 0
            ;;
        --help)
            echo "AirwavePBX v${VERSION} Management Script (Go Edition)"
            echo ""
            echo "Usage: $0 [OPTION]"
            echo ""
            echo "Options:"
            echo "  --install    Download and install AirwavePBX"
            echo "  --update     Check for and install updates"
            echo "  --status     Show system status"
            echo "  --start      Start all services"
            echo "  --stop       Stop all services"
            echo "  --restart    Restart all services"
            echo "  --backup     Create system backup"
            echo "  --help       Show this help"
            echo ""
            echo "Run without arguments to enter interactive mode."
            exit 0
            ;;
        *)
            print_error "Unknown option: $1"
            echo "Use --help for usage information"
            exit 1
            ;;
    esac
fi

# Interactive mode
check_root
show_main_menu