#!/bin/bash

# AirwavePBX v1.1.0 Installation Script (Go Edition)
# Target Platform: Debian 12 (Bookworm)
# Architecture: Single binary deployment with embedded SQLite

set -e

# Store the original script directory before any cd commands
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
BOLD='\033[1m'
NC='\033[0m' # No Color

# Installation configuration
AIRWAVE_USER="airwave"
AIRWAVE_GROUP="airwave"
AIRWAVE_HOME="/opt/airwavepbx"
ASTERISK_USER="asterisk"
ASTERISK_GROUP="asterisk"
ASTERISK_VERSION="21.5.0"
VERSION="1.1.0"

# Directories
INSTALL_DIR="/tmp/airwave-install"
LOG_DIR="/var/log/airwavepbx"
CONFIG_DIR="/etc/airwavepbx"
BACKUP_DIR="/var/backups/airwavepbx"
ASTERISK_CONFIG_DIR="/etc/asterisk"
RECORDINGS_DIR="/var/spool/asterisk/monitor"
DATA_DIR="/opt/airwavepbx/data"

# Files
INSTALL_LOG="${LOG_DIR}/install.log"
CREDENTIALS_FILE="/root/airwavepbx-credentials.txt"
BINARY_PATH="/opt/airwavepbx/airwavepbx"
SERVICE_FILE="/etc/systemd/system/airwavepbx.service"
ENV_FILE="/opt/airwavepbx/.env"

# Installation mode
INSTALL_MODE="fresh"  # fresh, update, repair, force
UPDATE_BACKUP_DIR=""
PRESERVE_DATA=false
DETECTED_ASTERISK=false
DETECTED_DB_CONFIG=false
EXISTING_AMI_PASSWORD=""
ADMIN_PASSWORD=""

# SSL Configuration
SSL_CERT="/etc/ssl/certs/airwavepbx.crt"
SSL_KEY="/etc/ssl/private/airwavepbx.key"
SSL_DAYS=3650

# Parse command line arguments
parse_arguments() {
    while [[ $# -gt 0 ]]; do
        case $1 in
            --update)
                INSTALL_MODE="update"
                shift
                ;;
            --repair)
                INSTALL_MODE="repair"
                shift
                ;;
            --force)
                INSTALL_MODE="force"
                shift
                ;;
            --help)
                show_help
                exit 0
                ;;
            *)
                echo "Unknown option: $1"
                show_help
                exit 1
                ;;
        esac
    done
}

show_help() {
    cat << EOF
AirwavePBX v1.1.0 Installation Script (Go Edition)

Usage: $0 [OPTIONS]

Options:
  --update    Update existing installation (preserves data)
  --repair    Repair existing installation 
  --force     Force fresh installation (removes existing data)
  --help      Show this help message

Examples:
  $0                    # Fresh installation
  $0 --update          # Update existing installation
  $0 --repair          # Repair installation issues
  $0 --force           # Force clean installation

EOF
}

# Utility functions
print_header() {
    echo ""
    echo -e "${BOLD}${BLUE}╔══════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${BOLD}${BLUE}║${NC}${BOLD}                    AirwavePBX v1.1.0 Installer                   ${NC}${BOLD}${BLUE}║${NC}"
    echo -e "${BOLD}${BLUE}║${NC}${BOLD}                        Go Edition                               ${NC}${BOLD}${BLUE}║${NC}"
    echo -e "${BOLD}${BLUE}╚══════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
}

print_step() {
    echo ""
    echo -e "${BOLD}${CYAN}▶ $1${NC}"
    echo "────────────────────────────────────────────────────────────────"
}

print_info() {
    echo -e "${BLUE}ℹ $1${NC}"
}

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

# Initialize logging
initialize_logging() {
    mkdir -p "$LOG_DIR"
    exec 19>&1
    exec > >(tee -a "$INSTALL_LOG")
    exec 2>&1
    echo "=== AirwavePBX v1.1.0 Installation Started: $(date) ===" >> "$INSTALL_LOG"
}

# Initialize credentials file
initialize_credentials_file() {
    print_info "Initializing credentials file..."
    cat > "$CREDENTIALS_FILE" << EOF
=== AIRWAVEPBX v1.1.0 INSTALLATION CREDENTIALS (Go Edition) ===
Generated on: $(date)
Installation Version: ${VERSION}

EOF
    chmod 600 "$CREDENTIALS_FILE"
    print_success "Credentials file initialized"
}

# Check if running as root
check_root() {
    if [[ $EUID -ne 0 ]]; then
        print_error "This script must be run as root (use sudo)"
        exit 1
    fi
}

# System requirements check
check_system_requirements() {
    print_step "Checking System Requirements"
    
    # Check OS
    if ! grep -q "Debian GNU/Linux 12" /etc/os-release 2>/dev/null; then
        print_warning "This installer is designed for Debian 12 (Bookworm)"
        print_info "Detected: $(cat /etc/os-release | grep PRETTY_NAME | cut -d'"' -f2)"
        read -p "Continue anyway? (y/N): " -n 1 -r
        echo
        if [[ ! $REPLY =~ ^[Yy]$ ]]; then
            exit 1
        fi
    fi
    
    # Check memory
    total_mem=$(free -m | awk '/^Mem:/{print $2}')
    if [ "$total_mem" -lt 1024 ]; then
        print_warning "System has ${total_mem}MB RAM. 2GB+ recommended for optimal performance."
    fi
    
    # Check disk space
    available_space=$(df / | awk 'NR==2{print $4}')
    if [ "$available_space" -lt 2097152 ]; then # 2GB in KB
        print_warning "Low disk space detected. 5GB+ recommended."
    fi
    
    print_success "System requirements check completed"
}

# Detect existing installation
detect_existing_installation() {
    print_step "Detecting Existing Installation"
    
    # Check for existing AirwavePBX
    if [ -f "$BINARY_PATH" ]; then
        PRESERVE_DATA=true
        print_info "Existing AirwavePBX v1.1.0 binary detected"
        
        # Get existing configuration
        if [ -f "$ENV_FILE" ]; then
            DETECTED_DB_CONFIG=true
            print_info "Existing configuration detected"
            
            # Extract AMI password if exists
            if grep -q "ASTERISK_PASS=" "$ENV_FILE"; then
                EXISTING_AMI_PASSWORD=$(grep "ASTERISK_PASS=" "$ENV_FILE" | cut -d'=' -f2 | tr -d '"')
                print_info "Existing AMI credentials detected"
            fi
        fi
    fi
    
    # Check for Asterisk
    if systemctl list-unit-files | grep -q "asterisk.service"; then
        DETECTED_ASTERISK=true
        print_info "Existing Asterisk installation detected"
    fi
    
    if [ "$PRESERVE_DATA" = true ]; then
        print_success "Existing installation detected - data preservation enabled"
    else
        print_info "No existing installation found - proceeding with fresh install"
    fi
}

# Install system dependencies
install_dependencies() {
    print_step "Installing System Dependencies"
    
    print_info "Updating package repository..."
    apt-get update -qq
    
    print_info "Installing essential packages..."
    apt-get install -y \
        wget \
        curl \
        gnupg \
        software-properties-common \
        apt-transport-https \
        ca-certificates \
        lsb-release \
        systemd \
        openssl \
        fail2ban \
        ufw \
        htop \
        nano \
        unzip \
        build-essential \
        pkg-config \
        libssl-dev \
        uuid-dev \
        libjansson-dev \
        libxml2-dev \
        libsqlite3-dev \
        libsrtp2-dev \
        libncurses5-dev \
        libncursesw5-dev \
        libcurl4-openssl-dev \
        libedit-dev \
        libvorbis-dev \
        libspandsp-dev \
        libgsm1-dev
    
    print_success "System dependencies installed"
}

# Get latest Asterisk version
get_latest_asterisk_version() {
    print_info "Detecting latest Asterisk version..."
    
    # Try to get the latest version from downloads page
    LATEST_VERSION=$(curl -s "https://downloads.asterisk.org/pub/telephony/asterisk/" | \
        grep -o 'asterisk-[0-9]\+\.[0-9]\+\.[0-9]\+\.tar\.gz' | \
        sed 's/asterisk-//g' | sed 's/\.tar\.gz//g' | \
        sort -V | tail -1)
    
    if [ -n "$LATEST_VERSION" ]; then
        ASTERISK_VERSION="$LATEST_VERSION"
        print_info "Latest Asterisk version detected: ${ASTERISK_VERSION}"
    else
        # Fallback to known good version
        ASTERISK_VERSION="21.8.0"
        print_warning "Could not detect latest version, using fallback: ${ASTERISK_VERSION}"
    fi
}

# Install and configure Asterisk
install_asterisk() {
    print_step "Installing Asterisk"
    
    # Get latest version
    get_latest_asterisk_version
    
    # Create asterisk user
    if ! id "$ASTERISK_USER" &>/dev/null; then
        print_info "Creating asterisk user..."
        useradd -r -s /bin/false -d /var/lib/asterisk -c "Asterisk PBX" "$ASTERISK_USER"
    fi
    
    # Check if Asterisk is already installed and working
    if [ "$DETECTED_ASTERISK" = true ] && asterisk -rx "core show version" &>/dev/null; then
        print_info "Asterisk is already installed and running"
        print_success "Skipping Asterisk compilation (15-20 minutes saved!)"
        return 0
    fi
    
    # Work in /usr/src directory
    cd /usr/src
    
    print_info "Downloading Asterisk ${ASTERISK_VERSION}..."
    if [ -f "asterisk-${ASTERISK_VERSION}.tar.gz" ]; then
        print_info "Using existing Asterisk source archive"
    else
        # Try multiple download sources
        DOWNLOAD_SUCCESS=false
        
        # Primary download URL
        if wget -O "asterisk-${ASTERISK_VERSION}.tar.gz" "https://downloads.asterisk.org/pub/telephony/asterisk/asterisk-${ASTERISK_VERSION}.tar.gz" 2>/dev/null; then
            DOWNLOAD_SUCCESS=true
        # Alternative download URL
        elif wget -O "asterisk-${ASTERISK_VERSION}.tar.gz" "https://github.com/asterisk/asterisk/archive/refs/tags/${ASTERISK_VERSION}.tar.gz" 2>/dev/null; then
            DOWNLOAD_SUCCESS=true
        # Try LTS version as fallback
        elif wget -O "asterisk-${ASTERISK_VERSION}.tar.gz" "https://downloads.asterisk.org/pub/telephony/asterisk/asterisk-21.8.0.tar.gz" 2>/dev/null; then
            ASTERISK_VERSION="21.8.0"
            print_warning "Using fallback version: ${ASTERISK_VERSION}"
            DOWNLOAD_SUCCESS=true
        fi
        
        if [ "$DOWNLOAD_SUCCESS" = false ]; then
            print_error "Failed to download Asterisk from all sources"
            return 1
        fi
        
        print_success "Downloaded Asterisk ${ASTERISK_VERSION}"
    fi
    
    print_info "Extracting Asterisk..."
    rm -rf "asterisk-${ASTERISK_VERSION}"
    if ! tar xzf "asterisk-${ASTERISK_VERSION}.tar.gz"; then
        print_error "Failed to extract Asterisk"
        return 1
    fi
    
    cd "asterisk-${ASTERISK_VERSION}"
    
    print_info "Configuring Asterisk build..."
    if ! ./configure --with-jansson-bundled --with-pjproject-bundled; then
        print_error "Failed to configure Asterisk"
        return 1
    fi
    
    print_info "Installing MP3 support..."
    contrib/scripts/get_mp3_source.sh || true
    
    print_info "Building Asterisk (this may take 15-20 minutes)..."
    if ! make -j$(nproc); then
        print_error "Failed to build Asterisk"
        return 1
    fi
    
    print_info "Installing Asterisk..."
    if ! make install; then
        print_error "Failed to install Asterisk"
        return 1
    fi
    
    print_info "Installing sample configuration..."
    make samples || true
    
    print_info "Installing init scripts..."
    make config || true
    
    # Set proper ownership
    chown -R "$ASTERISK_USER:$ASTERISK_USER" /etc/asterisk /var/lib/asterisk /var/log/asterisk /var/spool/asterisk /usr/lib/asterisk
    
    print_success "Asterisk ${ASTERISK_VERSION} installed successfully"
}

# Configure Asterisk for AirwavePBX
configure_asterisk() {
    print_step "Configuring Asterisk for AirwavePBX"
    
    # Generate AMI password if not preserved from existing installation
    if [ -z "$EXISTING_AMI_PASSWORD" ]; then
        AMI_PASSWORD=$(openssl rand -base64 32 | tr -d '/+=' | head -c 24)
    else
        AMI_PASSWORD="$EXISTING_AMI_PASSWORD"
        print_info "Using existing AMI password"
    fi
    
    # Create required directories
    mkdir -p /etc/asterisk
    mkdir -p /var/lib/asterisk
    mkdir -p /var/log/asterisk
    mkdir -p /var/spool/asterisk
    mkdir -p /var/run/asterisk
    
    # Set ownership
    chown -R "$ASTERISK_USER:$ASTERISK_USER" /etc/asterisk /var/lib/asterisk /var/log/asterisk /var/spool/asterisk /var/run/asterisk
    
    # Configure AMI
    cat > "$ASTERISK_CONFIG_DIR/manager.conf" << EOF
[general]
enabled = yes
port = 5038
bindaddr = 127.0.0.1
displayconnects = no
timestampevents = yes

[airwaveami]
secret = ${AMI_PASSWORD}
deny = 0.0.0.0/0.0.0.0
permit = 127.0.0.1/255.255.255.255
read = all
write = all
EOF
    
    # Configure basic extensions
    cat > "$ASTERISK_CONFIG_DIR/extensions.conf" << EOF
[general]
static=yes
writeprotect=no
clearglobalvars=no

[globals]

[default]
exten => _X.,1,Dial(SIP/\${EXTEN})
same => n,Hangup()

[internal]
exten => _XXX,1,Dial(SIP/\${EXTEN},30)
same => n,Voicemail(\${EXTEN}@default,u)
same => n,Hangup()
EOF
    
    # Configure SIP
    cat > "$ASTERISK_CONFIG_DIR/pjsip.conf" << EOF
[global]
type=global
endpoint_identifier_order=ip,username,anonymous

[transport-udp]
type=transport
protocol=udp
bind=0.0.0.0:5060

[transport-tcp]
type=transport
protocol=tcp
bind=0.0.0.0:5060

[anonymous]
type=endpoint
context=default
disallow=all
allow=ulaw
allow=alaw
auth_type=userpass
EOF
    
    # Set proper ownership
    chown -R "$ASTERISK_USER:$ASTERISK_USER" "$ASTERISK_CONFIG_DIR"
    
    # Save AMI password to credentials
    echo "ASTERISK_AMI_PASSWORD=${AMI_PASSWORD}" >> "$CREDENTIALS_FILE"
    
    print_success "Asterisk configuration completed"
}

# Create AirwavePBX user and directories
create_airwave_user() {
    print_step "Creating AirwavePBX System User"
    
    # Create airwave user
    if ! id "$AIRWAVE_USER" &>/dev/null; then
        print_info "Creating airwave system user..."
        useradd -r -s /bin/bash -d "$AIRWAVE_HOME" -c "AirwavePBX System User" "$AIRWAVE_USER"
    fi
    
    # Create directories
    mkdir -p "$AIRWAVE_HOME"
    mkdir -p "$DATA_DIR"
    mkdir -p "$CONFIG_DIR"
    mkdir -p "$BACKUP_DIR"
    mkdir -p "$RECORDINGS_DIR"
    
    # Set ownership
    chown -R "$AIRWAVE_USER:$AIRWAVE_GROUP" "$AIRWAVE_HOME"
    chown -R "$AIRWAVE_USER:$AIRWAVE_GROUP" "$DATA_DIR"
    chown -R "$AIRWAVE_USER:$AIRWAVE_GROUP" "$CONFIG_DIR"
    chown -R "$AIRWAVE_USER:$AIRWAVE_GROUP" "$BACKUP_DIR"
    
    print_success "AirwavePBX user and directories created"
}

# Install AirwavePBX binary
install_airwavepbx_binary() {
    print_step "Installing AirwavePBX v1.1.0 Binary"
    
    # Copy binary from build directory
    if [ -f "$SCRIPT_DIR/airwavepbx" ]; then
        print_info "Installing AirwavePBX binary..."
        cp "$SCRIPT_DIR/airwavepbx" "$BINARY_PATH"
        chmod 755 "$BINARY_PATH"
        chown "$AIRWAVE_USER:$AIRWAVE_GROUP" "$BINARY_PATH"
        print_success "AirwavePBX binary installed"
    else
        print_error "AirwavePBX binary not found in $SCRIPT_DIR"
        print_info "Please ensure the binary is built and present in the installation directory"
        exit 1
    fi
}

# Generate SSL certificates
generate_ssl_certificates() {
    print_step "Generating SSL Certificates"
    
    # Check if certificates already exist
    if [ -f "$SSL_CERT" ] && [ -f "$SSL_KEY" ] && [ "$INSTALL_MODE" != "force" ]; then
        print_info "SSL certificates already exist, skipping generation"
        return 0
    fi
    
    # Create directories
    mkdir -p /etc/ssl/certs
    mkdir -p /etc/ssl/private
    
    # Ask user for certificate type
    echo ""
    echo "SSL Certificate Options:"
    echo "1) Let's Encrypt (recommended for production)"
    echo "2) Self-signed certificate (for testing)"
    echo ""
    read -p "Choose certificate type [1-2]: " cert_choice
    
    case $cert_choice in
        1)
            generate_letsencrypt_certificate
            ;;
        2|*)
            generate_selfsigned_certificate
            ;;
    esac
    
    echo "SSL_CERTIFICATE_PATH=${SSL_CERT}" >> "$CREDENTIALS_FILE"
    echo "SSL_PRIVATE_KEY_PATH=${SSL_KEY}" >> "$CREDENTIALS_FILE"
}

generate_letsencrypt_certificate() {
    print_info "Setting up Let's Encrypt certificate..."
    
    # Install certbot
    apt-get update -qq
    apt-get install -y certbot
    
    # Get domain name
    HOSTNAME=$(hostname -f)
    echo ""
    echo "Current hostname: $HOSTNAME"
    read -p "Enter domain name for SSL certificate [$HOSTNAME]: " DOMAIN
    DOMAIN=${DOMAIN:-$HOSTNAME}
    
    # Get email for notifications
    read -p "Enter email address for Let's Encrypt notifications: " EMAIL
    
    if [ -z "$EMAIL" ]; then
        print_error "Email address is required for Let's Encrypt"
        generate_selfsigned_certificate
        return
    fi
    
    # Stop any running web servers temporarily
    systemctl stop nginx 2>/dev/null || true
    systemctl stop apache2 2>/dev/null || true
    
    # Generate certificate
    print_info "Requesting Let's Encrypt certificate for $DOMAIN..."
    if certbot certonly --standalone --non-interactive --agree-tos --email "$EMAIL" -d "$DOMAIN"; then
        # Copy certificates to our standard location
        cp "/etc/letsencrypt/live/$DOMAIN/fullchain.pem" "$SSL_CERT"
        cp "/etc/letsencrypt/live/$DOMAIN/privkey.pem" "$SSL_KEY"
        
        # Set permissions
        chmod 644 "$SSL_CERT"
        chmod 600 "$SSL_KEY"
        chown root:root "$SSL_CERT" "$SSL_KEY"
        
        # Create renewal script
        cat > /etc/cron.d/letsencrypt-airwavepbx << EOF
# Renew Let's Encrypt certificates for AirwavePBX
0 3 * * * root certbot renew --quiet --post-hook "cp /etc/letsencrypt/live/$DOMAIN/fullchain.pem $SSL_CERT && cp /etc/letsencrypt/live/$DOMAIN/privkey.pem $SSL_KEY && systemctl restart airwavepbx"
EOF
        
        print_success "Let's Encrypt certificate generated successfully"
        echo "LETSENCRYPT_DOMAIN=${DOMAIN}" >> "$CREDENTIALS_FILE"
        echo "LETSENCRYPT_EMAIL=${EMAIL}" >> "$CREDENTIALS_FILE"
    else
        print_warning "Let's Encrypt certificate generation failed, falling back to self-signed"
        generate_selfsigned_certificate
    fi
}

generate_selfsigned_certificate() {
    print_info "Generating self-signed SSL certificate..."
    
    # Generate private key
    openssl genrsa -out "$SSL_KEY" 4096
    
    # Generate certificate
    openssl req -new -x509 -key "$SSL_KEY" -out "$SSL_CERT" -days "$SSL_DAYS" -subj "/C=US/ST=Local/L=Local/O=AirwavePBX/OU=IT/CN=$(hostname -f)"
    
    # Set permissions
    chmod 644 "$SSL_CERT"
    chmod 600 "$SSL_KEY"
    chown root:root "$SSL_CERT" "$SSL_KEY"
    
    print_success "Self-signed SSL certificate generated successfully"
}

# Create configuration file
create_configuration() {
    print_step "Creating AirwavePBX Configuration"
    
    # Generate admin password if not preserved
    if [ -z "$ADMIN_PASSWORD" ]; then
        ADMIN_PASSWORD=$(openssl rand -base64 16 | tr -d '/+=' | head -c 12)
    fi
    
    # Generate JWT secret
    JWT_SECRET=$(openssl rand -base64 64 | tr -d '/+=' | head -c 64)
    
    # Create environment file
    cat > "$ENV_FILE" << EOF
# AirwavePBX v1.1.0 Configuration (Go Edition)
# Generated on: $(date)

# Server Configuration
PORT=8443
SSL_CERT=${SSL_CERT}
SSL_KEY=${SSL_KEY}

# Database Configuration (SQLite)
DB_PATH=${DATA_DIR}/airwavepbx.db

# Asterisk AMI Configuration
ASTERISK_HOST=127.0.0.1
ASTERISK_PORT=5038
ASTERISK_USER=airwaveami
ASTERISK_PASS=${AMI_PASSWORD}

# Security
JWT_SECRET=${JWT_SECRET}
ADMIN_PASSWORD=${ADMIN_PASSWORD}

# Logging
LOG_LEVEL=info
EOF
    
    # Set permissions
    chmod 600 "$ENV_FILE"
    chown "$AIRWAVE_USER:$AIRWAVE_GROUP" "$ENV_FILE"
    
    # Save admin credentials
    echo "" >> "$CREDENTIALS_FILE"
    echo "=== WEB INTERFACE CREDENTIALS ===" >> "$CREDENTIALS_FILE"
    echo "Admin Username: admin" >> "$CREDENTIALS_FILE"
    echo "Admin Password: ${ADMIN_PASSWORD}" >> "$CREDENTIALS_FILE"
    echo "Web Interface: https://$(hostname -I | awk '{print $1}'):8443" >> "$CREDENTIALS_FILE"
    
    print_success "Configuration file created"
}

# Create systemd service
create_systemd_service() {
    print_step "Creating Systemd Service"
    
    cat > "$SERVICE_FILE" << EOF
[Unit]
Description=AirwavePBX v1.1.0 (Go Edition)
Documentation=https://docs.airwavepbx.com
After=network.target asterisk.service
Wants=asterisk.service
Requires=network.target

[Service]
Type=simple
User=${AIRWAVE_USER}
Group=${AIRWAVE_GROUP}
WorkingDirectory=${AIRWAVE_HOME}
ExecStart=${BINARY_PATH}
EnvironmentFile=${ENV_FILE}
Restart=always
RestartSec=10
StandardOutput=journal
StandardError=journal
SyslogIdentifier=airwavepbx

# Security settings
NoNewPrivileges=true
PrivateTmp=true
ProtectSystem=strict
ProtectHome=true
ReadWritePaths=${AIRWAVE_HOME} ${DATA_DIR} ${LOG_DIR}

# Resource limits
LimitNOFILE=65536
LimitNPROC=4096

[Install]
WantedBy=multi-user.target
EOF
    
    # Reload systemd and enable service
    systemctl daemon-reload
    systemctl enable airwavepbx.service
    
    print_success "Systemd service created and enabled"
}

# Configure firewall
configure_firewall() {
    print_step "Configuring Firewall"
    
    print_info "Setting up UFW firewall rules..."
    
    # Reset UFW to defaults
    ufw --force reset > /dev/null 2>&1
    
    # Set default policies
    ufw default deny incoming > /dev/null 2>&1
    ufw default allow outgoing > /dev/null 2>&1
    
    # Allow SSH
    ufw allow ssh > /dev/null 2>&1
    
    # Allow AirwavePBX HTTPS
    ufw allow 8443/tcp comment "AirwavePBX HTTPS" > /dev/null 2>&1
    
    # Allow SIP
    ufw allow 5060/udp comment "SIP UDP" > /dev/null 2>&1
    ufw allow 5060/tcp comment "SIP TCP" > /dev/null 2>&1
    
    # Allow RTP
    ufw allow 10000:20000/udp comment "RTP" > /dev/null 2>&1
    
    # Enable firewall
    ufw --force enable > /dev/null 2>&1
    
    print_success "Firewall configured successfully"
}

# Start services
start_services() {
    print_step "Starting Services"
    
    # Start Asterisk
    print_info "Starting Asterisk..."
    systemctl restart asterisk
    systemctl enable asterisk
    
    # Wait for Asterisk to start
    sleep 5
    
    # Start AirwavePBX
    print_info "Starting AirwavePBX..."
    systemctl restart airwavepbx
    
    # Wait for services to start
    sleep 10
    
    # Check service status
    if systemctl is-active --quiet asterisk; then
        print_success "Asterisk started successfully"
    else
        print_warning "Asterisk may have startup issues"
    fi
    
    if systemctl is-active --quiet airwavepbx; then
        print_success "AirwavePBX started successfully"
    else
        print_warning "AirwavePBX may have startup issues"
    fi
}

# Create admin user in database
create_admin_user() {
    print_step "Creating Admin User"
    
    # Wait for AirwavePBX to initialize database
    sleep 5
    
    # The Go application will create the admin user automatically
    # We just need to document the credentials
    print_info "Admin user will be created automatically on first startup"
    print_success "Admin credentials saved to ${CREDENTIALS_FILE}"
}

# Installation completion
installation_complete() {
    print_step "Installation Complete"
    
    echo ""
    echo -e "${BOLD}${GREEN}╔══════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${BOLD}${GREEN}║${NC}${BOLD}                    🎉 INSTALLATION COMPLETE! 🎉                 ${NC}${BOLD}${GREEN}║${NC}"
    echo -e "${BOLD}${GREEN}║${NC}${BOLD}                      AirwavePBX v1.1.0                          ${NC}${BOLD}${GREEN}║${NC}"
    echo -e "${BOLD}${GREEN}║${NC}${BOLD}                Your PBX is Ready for Production                  ${NC}${BOLD}${GREEN}║${NC}"
    echo -e "${BOLD}${GREEN}╚══════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
    
    # Display access information
    SERVER_IP=$(hostname -I | awk '{print $1}')
    echo -e "${BOLD}${CYAN}🌐 Web Interface:${NC} https://${SERVER_IP}:8443"
    echo -e "${BOLD}${CYAN}👤 Admin Username:${NC} admin"
    echo -e "${BOLD}${CYAN}🔑 Admin Password:${NC} ${ADMIN_PASSWORD}"
    echo ""
    
    # Display important information
    echo -e "${BOLD}${YELLOW}📋 Important Information:${NC}"
    echo -e "${YELLOW}   • All credentials saved to: ${CREDENTIALS_FILE}${NC}"
    echo -e "${YELLOW}   • Installation logs: ${INSTALL_LOG}${NC}"
    echo -e "${YELLOW}   • Service management: systemctl [start|stop|restart] airwavepbx${NC}"
    echo -e "${YELLOW}   • Configuration file: ${ENV_FILE}${NC}"
    echo ""
    
    # Reboot reminder
    echo -e "${BOLD}${RED}╔══════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${BOLD}${RED}║ IMPORTANT: Reboot your server for a clean startup of all system ║${NC}"
    echo -e "${BOLD}${RED}║            services.                                             ║${NC}"
    echo -e "${BOLD}${RED}╚══════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
    
    echo -e "${BOLD}${GREEN}🚀 AirwavePBX v1.1.0 (Go Edition) installation completed successfully!${NC}"
    echo ""
}

# Error handling
handle_error() {
    print_error "Installation failed at step: $1"
    print_info "Check the installation log: $INSTALL_LOG"
    exit 1
}

# Main installation function
main() {
    # Parse arguments
    parse_arguments "$@"
    
    # Initialize
    print_header
    check_root
    initialize_logging
    initialize_credentials_file
    
    # Installation steps
    check_system_requirements || handle_error "system_requirements"
    detect_existing_installation || handle_error "detect_existing"
    install_dependencies || handle_error "dependencies"
    install_asterisk || handle_error "asterisk_install"
    configure_asterisk || handle_error "asterisk_config"
    create_airwave_user || handle_error "user_creation"
    install_airwavepbx_binary || handle_error "binary_install"
    generate_ssl_certificates || handle_error "ssl_certificates"
    create_configuration || handle_error "configuration"
    create_systemd_service || handle_error "systemd_service"
    configure_firewall || handle_error "firewall"
    start_services || handle_error "start_services"
    create_admin_user || handle_error "admin_user"
    installation_complete
}

# Run main function
main "$@"