// Extensions Page

async function loadExtensionsPage() {
    const contentDiv = document.getElementById('page-content');
    
    contentDiv.innerHTML = `
        <div class="page-header">
            <h1 class="page-title">Extensions</h1>
            <div class="page-actions">
                <button class="btn btn-primary" onclick="showAddExtension()">
                    <i class="fap fa-plus"></i>
                    <span>Add Extension</span>
                </button>
            </div>
        </div>
        
        <div class="glass-card">
            <div id="extensions-table-container">
                <div class="loading">Loading extensions...</div>
            </div>
        </div>
    `;
    
    await loadExtensions();
}

async function loadExtensions() {
    try {
        const extensions = await API.getExtensions();
        renderExtensionsTable(extensions);
    } catch (error) {
        showNotification('Error', 'Failed to load extensions', 'error');
    }
}

function renderExtensionsTable(extensions) {
    const container = document.getElementById('extensions-table-container');
    
    if (extensions.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <i class="fap fa-users" style="font-size: 48px; color: var(--text-tertiary);"></i>
                <p class="mt-3">No extensions configured</p>
                <button class="btn btn-primary mt-3" onclick="showAddExtension()">
                    <i class="fap fa-plus"></i>
                    <span>Add First Extension</span>
                </button>
            </div>
        `;
        return;
    }
    
    container.innerHTML = `
        <table class="glass-table">
            <thead>
                <tr>
                    <th>Extension</th>
                    <th>Name</th>
                    <th>Caller ID</th>
                    <th>Voicemail</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                ${extensions.map(ext => `
                    <tr>
                        <td>
                            <span class="extension-number">${ext.extension}</span>
                        </td>
                        <td>${ext.name}</td>
                        <td>
                            <div>${ext.callerid_name}</div>
                            <small class="text-secondary">${ext.callerid_number}</small>
                        </td>
                        <td>
                            <span class="badge ${ext.voicemail_enabled ? 'success' : 'secondary'}">
                                ${ext.voicemail_enabled ? 'Enabled' : 'Disabled'}
                            </span>
                        </td>
                        <td>
                            <div class="status-indicator ${ext.enabled ? 'success' : 'danger'}"></div>
                            ${ext.enabled ? 'Active' : 'Inactive'}
                        </td>
                        <td>
                            <div class="action-buttons">
                                <button class="btn-icon" onclick="editExtension(${ext.id})" title="Edit">
                                    <i class="fap fa-edit"></i>
                                </button>
                                <button class="btn-icon" onclick="confirmDeleteExtension(${ext.id}, '${ext.extension}')" title="Delete">
                                    <i class="fap fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                `).join('')}
            </tbody>
        </table>
    `;
}

function showAddExtension() {
    showExtensionForm('Add Extension', null);
}

async function editExtension(id) {
    try {
        const extension = await API.getExtension(id);
        showExtensionForm('Edit Extension', extension);
    } catch (error) {
        showNotification('Error', 'Failed to load extension details', 'error');
    }
}

function showExtensionForm(title, extension) {
    const isEdit = !!extension;
    
    showModal(title, `
        <form id="extension-form" onsubmit="handleExtensionSubmit(event, ${isEdit ? extension.id : 'null'})">
            <div class="form-group">
                <label>Extension Number *</label>
                <div class="glass-input">
                    <i class="fap fa-hashtag input-icon"></i>
                    <input type="text" name="extension" value="${extension?.extension || ''}" 
                           placeholder="e.g., 1001" required ${isEdit ? 'readonly' : ''}>
                </div>
            </div>
            
            <div class="form-group">
                <label>Name *</label>
                <div class="glass-input">
                    <i class="fap fa-user input-icon"></i>
                    <input type="text" name="name" value="${extension?.name || ''}" 
                           placeholder="e.g., John Doe" required>
                </div>
            </div>
            
            <div class="form-group">
                <label>Password ${isEdit ? '(leave blank to keep current)' : '*'}</label>
                <div class="glass-input">
                    <i class="fap fa-lock input-icon"></i>
                    <input type="password" name="password" placeholder="Strong password" ${isEdit ? '' : 'required'}>
                </div>
            </div>
            
            <div class="form-group">
                <label>Caller ID Name</label>
                <div class="glass-input">
                    <i class="fap fa-id-card input-icon"></i>
                    <input type="text" name="callerid_name" value="${extension?.callerid_name || ''}" 
                           placeholder="Display name">
                </div>
            </div>
            
            <div class="form-group">
                <label>Caller ID Number</label>
                <div class="glass-input">
                    <i class="fap fa-phone input-icon"></i>
                    <input type="text" name="callerid_number" value="${extension?.callerid_number || ''}" 
                           placeholder="Phone number">
                </div>
            </div>
            
            <div class="form-group">
                <label class="checkbox-label">
                    <input type="checkbox" name="voicemail_enabled" ${extension?.voicemail_enabled ? 'checked' : ''}>
                    <span>Enable Voicemail</span>
                </label>
            </div>
            
            <div class="form-group" id="voicemail-password-group" style="display: ${extension?.voicemail_enabled ? 'block' : 'none'}">
                <label>Voicemail Password</label>
                <div class="glass-input">
                    <i class="fap fa-key input-icon"></i>
                    <input type="text" name="voicemail_password" value="${extension?.voicemail_password || ''}" 
                           placeholder="Numeric password">
                </div>
            </div>
            
            <div class="form-group">
                <label class="checkbox-label">
                    <input type="checkbox" name="enabled" ${extension?.enabled !== false ? 'checked' : ''}>
                    <span>Enable Extension</span>
                </label>
            </div>
        </form>
    `, [
        {
            label: 'Cancel',
            onclick: 'closeModal()'
        },
        {
            label: isEdit ? 'Update' : 'Create',
            class: 'btn-primary',
            icon: isEdit ? 'save' : 'plus',
            onclick: "document.getElementById('extension-form').submit()"
        }
    ]);
    
    // Toggle voicemail password field
    document.querySelector('input[name="voicemail_enabled"]').addEventListener('change', (e) => {
        document.getElementById('voicemail-password-group').style.display = 
            e.target.checked ? 'block' : 'none';
    });
}

async function handleExtensionSubmit(event, extensionId) {
    event.preventDefault();
    const formData = new FormData(event.target);
    
    const data = {
        extension: formData.get('extension'),
        name: formData.get('name'),
        callerid_name: formData.get('callerid_name') || formData.get('name'),
        callerid_number: formData.get('callerid_number') || formData.get('extension'),
        voicemail_enabled: formData.get('voicemail_enabled') === 'on',
        voicemail_password: formData.get('voicemail_password'),
        enabled: formData.get('enabled') === 'on'
    };
    
    const password = formData.get('password');
    if (password) {
        data.password = password;
    }
    
    try {
        if (extensionId) {
            await API.updateExtension(extensionId, data);
            showNotification('Success', 'Extension updated successfully', 'success');
        } else {
            await API.createExtension(data);
            showNotification('Success', 'Extension created successfully', 'success');
        }
        closeModal();
        await loadExtensions();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

function confirmDeleteExtension(id, extension) {
    showModal('Delete Extension', `
        <p>Are you sure you want to delete extension <strong>${extension}</strong>?</p>
        <p class="text-warning mt-3">
            <i class="fap fa-exclamation-triangle"></i>
            This action cannot be undone.
        </p>
    `, [
        {
            label: 'Cancel',
            onclick: 'closeModal()'
        },
        {
            label: 'Delete',
            class: 'btn-danger',
            icon: 'trash',
            onclick: `deleteExtension(${id})`
        }
    ]);
}

async function deleteExtension(id) {
    try {
        await API.deleteExtension(id);
        closeModal();
        showNotification('Success', 'Extension deleted successfully', 'success');
        await loadExtensions();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

// Add extensions-specific styles
const extensionsStyles = `
<style>
.extension-number {
    font-weight: 600;
    font-size: 16px;
    color: var(--primary-color);
}

.checkbox-label {
    display: flex;
    align-items: center;
    cursor: pointer;
    user-select: none;
}

.checkbox-label input[type="checkbox"] {
    margin-right: var(--spacing-sm);
    cursor: pointer;
}

.form-group label {
    display: block;
    margin-bottom: var(--spacing-sm);
    font-weight: 500;
    font-size: 14px;
}

.form-group {
    margin-bottom: var(--spacing-md);
}

#extension-form .glass-input input[readonly] {
    background: var(--glass-bg);
    cursor: not-allowed;
}
</style>
`;

if (!document.getElementById('extensions-styles')) {
    const styleElement = document.createElement('div');
    styleElement.id = 'extensions-styles';
    styleElement.innerHTML = extensionsStyles;
    document.head.appendChild(styleElement);
}