// Call History Page

let currentPage = 0;
const pageSize = 50;

async function loadHistoryPage() {
    const contentDiv = document.getElementById('page-content');
    
    contentDiv.innerHTML = `
        <div class="page-header">
            <h1 class="page-title">Call History</h1>
            <div class="page-actions">
                <div class="glass-input">
                    <i class="fap fa-search input-icon"></i>
                    <input type="text" id="history-search" placeholder="Search calls..." onkeyup="searchCallHistory(this.value)">
                </div>
                <button class="btn" onclick="exportCallHistory()">
                    <i class="fap fa-download"></i>
                    <span>Export</span>
                </button>
            </div>
        </div>
        
        <div class="glass-card">
            <div id="history-table-container">
                <div class="loading">Loading call history...</div>
            </div>
            <div id="history-pagination" class="pagination"></div>
        </div>
    `;
    
    // Load initial data
    await loadCallHistory();
}

async function loadCallHistory(page = 0) {
    currentPage = page;
    const offset = page * pageSize;
    
    try {
        const { calls, total } = await API.getCallHistory(pageSize, offset);
        renderCallHistoryTable(calls);
        renderPagination(total);
    } catch (error) {
        showNotification('Error', 'Failed to load call history', 'error');
    }
}

function renderCallHistoryTable(calls) {
    const container = document.getElementById('history-table-container');
    
    if (calls.length === 0) {
        container.innerHTML = '<p class="text-center p-4">No call history found</p>';
        return;
    }
    
    container.innerHTML = `
        <table class="glass-table">
            <thead>
                <tr>
                    <th>Date/Time</th>
                    <th>Direction</th>
                    <th>Caller</th>
                    <th>Destination</th>
                    <th>Duration</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                ${calls.map(call => `
                    <tr>
                        <td>${formatDateTime(call.start_time)}</td>
                        <td>
                            <span class="direction-badge ${call.direction}">
                                <i class="fap fa-${call.direction === 'inbound' ? 'phone-arrow-down-left' : 'phone-arrow-up-right'}"></i>
                                ${call.direction}
                            </span>
                        </td>
                        <td>
                            <div>${call.caller_id_name || 'Unknown'}</div>
                            <small class="text-secondary">${formatPhoneNumber(call.caller_id_number)}</small>
                        </td>
                        <td>${formatPhoneNumber(call.destination_number)}</td>
                        <td>${formatDuration(call.duration)}</td>
                        <td>
                            <span class="badge ${getCallStatusClass(call.disposition)}">
                                ${call.disposition}
                            </span>
                        </td>
                        <td>
                            <div class="action-buttons">
                                ${call.recording_path ? `
                                    <button class="btn-icon" onclick="playRecording('${call.recording_path}')" title="Play Recording">
                                        <i class="fap fa-play-circle"></i>
                                    </button>
                                ` : ''}
                                <button class="btn-icon" onclick="showCallDetails('${call.id}')" title="View Details">
                                    <i class="fap fa-info-circle"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                `).join('')}
            </tbody>
        </table>
    `;
}

function renderPagination(total) {
    const totalPages = Math.ceil(total / pageSize);
    const container = document.getElementById('history-pagination');
    
    if (totalPages <= 1) {
        container.innerHTML = '';
        return;
    }
    
    const pages = [];
    
    // Previous button
    pages.push(`
        <button class="pagination-btn" onclick="loadCallHistory(${currentPage - 1})" ${currentPage === 0 ? 'disabled' : ''}>
            <i class="fap fa-chevron-left"></i>
        </button>
    `);
    
    // Page numbers
    for (let i = 0; i < totalPages; i++) {
        if (
            i === 0 || 
            i === totalPages - 1 || 
            (i >= currentPage - 2 && i <= currentPage + 2)
        ) {
            pages.push(`
                <button class="pagination-btn ${i === currentPage ? 'active' : ''}" onclick="loadCallHistory(${i})">
                    ${i + 1}
                </button>
            `);
        } else if (i === currentPage - 3 || i === currentPage + 3) {
            pages.push('<span class="pagination-ellipsis">...</span>');
        }
    }
    
    // Next button
    pages.push(`
        <button class="pagination-btn" onclick="loadCallHistory(${currentPage + 1})" ${currentPage === totalPages - 1 ? 'disabled' : ''}>
            <i class="fap fa-chevron-right"></i>
        </button>
    `);
    
    container.innerHTML = pages.join('');
}

async function searchCallHistory(query) {
    // Debounced search implementation
    if (window.searchTimeout) {
        clearTimeout(window.searchTimeout);
    }
    
    window.searchTimeout = setTimeout(async () => {
        // For now, filter client-side
        // In production, this would be a server-side search
        await loadCallHistory(0);
        
        if (query) {
            const rows = document.querySelectorAll('#history-table-container tbody tr');
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(query.toLowerCase()) ? '' : 'none';
            });
        }
    }, 300);
}

function playRecording(path) {
    showModal('Call Recording', `
        <audio controls autoplay style="width: 100%;">
            <source src="/recordings/${path}" type="audio/wav">
            Your browser does not support the audio element.
        </audio>
    `, [
        {
            label: 'Close',
            onclick: 'closeModal()'
        }
    ]);
}

async function showCallDetails(callId) {
    // In a real implementation, this would fetch full call details
    showModal('Call Details', `
        <div class="call-details">
            <p>Call ID: ${callId}</p>
            <p>Additional call details would be shown here...</p>
        </div>
    `, [
        {
            label: 'Close',
            onclick: 'closeModal()'
        }
    ]);
}

async function exportCallHistory() {
    try {
        // Get all calls for export
        const { calls } = await API.getCallHistory(1000, 0);
        
        // Convert to CSV
        const csv = [
            ['Date/Time', 'Direction', 'Caller Name', 'Caller Number', 'Destination', 'Duration', 'Status'],
            ...calls.map(call => [
                new Date(call.start_time).toLocaleString(),
                call.direction,
                call.caller_id_name || 'Unknown',
                call.caller_id_number,
                call.destination_number,
                formatDuration(call.duration),
                call.disposition
            ])
        ].map(row => row.map(cell => `"${cell}"`).join(',')).join('\n');
        
        // Download
        const blob = new Blob([csv], { type: 'text/csv' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `call-history-${new Date().toISOString().split('T')[0]}.csv`;
        a.click();
        URL.revokeObjectURL(url);
        
        showNotification('Export Complete', 'Call history exported successfully', 'success');
    } catch (error) {
        showNotification('Error', 'Failed to export call history', 'error');
    }
}

// Add history-specific styles
const historyStyles = `
<style>
.direction-badge {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    padding: 2px 8px;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 500;
    text-transform: capitalize;
}

.direction-badge.inbound {
    background: rgba(52, 199, 89, 0.2);
    color: var(--success-color);
}

.direction-badge.outbound {
    background: rgba(0, 122, 255, 0.2);
    color: var(--primary-color);
}

.action-buttons {
    display: flex;
    gap: var(--spacing-xs);
}

.pagination {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: var(--spacing-xs);
    margin-top: var(--spacing-lg);
}

.pagination-btn {
    min-width: 36px;
    height: 36px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: var(--glass-bg);
    border: 1px solid var(--glass-border);
    border-radius: var(--radius-sm);
    color: var(--text-primary);
    cursor: pointer;
    transition: all var(--transition-fast);
}

.pagination-btn:hover:not(:disabled) {
    background: var(--glass-border);
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.pagination-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.pagination-ellipsis {
    padding: 0 var(--spacing-sm);
    color: var(--text-tertiary);
}

.call-details {
    font-size: 14px;
    line-height: 1.8;
}
</style>
`;

if (!document.getElementById('history-styles')) {
    const styleElement = document.createElement('div');
    styleElement.id = 'history-styles';
    styleElement.innerHTML = historyStyles;
    document.head.appendChild(styleElement);
}