package websocket

import (
	"encoding/json"
	"fmt"
	"log"
	
	"airwavepbx/internal/ami"
	"github.com/gofiber/fiber/v2"
	"github.com/gofiber/websocket/v2"
)

// Message types
const (
	MessageTypeEvent       = "event"
	MessageTypeCallUpdate  = "call_update"
	MessageTypeCallRinging = "call_ringing"
	MessageTypeCallAnswered = "call_answered"
	MessageTypeCallEnded   = "call_ended"
	MessageTypeStats       = "stats"
	MessageTypePing        = "ping"
	MessageTypePong        = "pong"
)

// Client represents a WebSocket client
type Client struct {
	conn   *websocket.Conn
	send   chan []byte
	hub    *Hub
	id     string
}

// Hub maintains active WebSocket connections
type Hub struct {
	clients    map[*Client]bool
	broadcast  chan []byte
	register   chan *Client
	unregister chan *Client
}

// Message represents a WebSocket message
type Message struct {
	Type string      `json:"type"`
	Data interface{} `json:"data"`
}

// CallEvent represents a call-related event
type CallEvent struct {
	UUID           string `json:"uuid"`
	CallerIDName   string `json:"caller_id_name"`
	CallerIDNumber string `json:"caller_id_number"`
	Destination    string `json:"destination"`
	Line           int    `json:"line"`
	State          string `json:"state"`
	Duration       int    `json:"duration"`
}

// NewHub creates a new WebSocket hub
func NewHub() *Hub {
	return &Hub{
		clients:    make(map[*Client]bool),
		broadcast:  make(chan []byte, 256),
		register:   make(chan *Client),
		unregister: make(chan *Client),
	}
}

// Run starts the hub's event loop
func (h *Hub) Run() {
	for {
		select {
		case client := <-h.register:
			h.clients[client] = true
			log.Printf("WebSocket client connected: %s", client.id)
			
			// Send current stats on connect
			h.sendStats()
			
		case client := <-h.unregister:
			if _, ok := h.clients[client]; ok {
				delete(h.clients, client)
				close(client.send)
				log.Printf("WebSocket client disconnected: %s", client.id)
			}
			
		case message := <-h.broadcast:
			for client := range h.clients {
				select {
				case client.send <- message:
				default:
					// Client's send channel is full, close it
					delete(h.clients, client)
					close(client.send)
				}
			}
		}
	}
}

// BroadcastEvent sends an Asterisk AMI event to all clients
func (h *Hub) BroadcastEvent(event ami.Event) {
	// Convert AMI event to our format
	var msg Message
	
	switch event.Name {
	case "Newchannel":
		// Extract UUID from channel variable if set, otherwise use channel name
		uuid := event.Headers["Uniqueid"]
		if airwaveUUID := event.Headers["Variable_AIRWAVE_UUID"]; airwaveUUID != "" {
			uuid = airwaveUUID
		}
		
		msg = Message{
			Type: MessageTypeCallRinging,
			Data: CallEvent{
				UUID:           uuid,
				CallerIDName:   event.Headers["CallerIDName"],
				CallerIDNumber: event.Headers["CallerIDNum"],
				Destination:    event.Headers["Exten"],
				State:          "ringing",
			},
		}
		
	case "Newstate":
		if event.Headers["ChannelStateDesc"] == "Up" {
			uuid := event.Headers["Uniqueid"]
			if airwaveUUID := event.Headers["Variable_AIRWAVE_UUID"]; airwaveUUID != "" {
				uuid = airwaveUUID
			}
			
			msg = Message{
				Type: MessageTypeCallAnswered,
				Data: CallEvent{
					UUID:  uuid,
					State: "answered",
				},
			}
		}
		
	case "Hangup":
		uuid := event.Headers["Uniqueid"]
		if airwaveUUID := event.Headers["Variable_AIRWAVE_UUID"]; airwaveUUID != "" {
			uuid = airwaveUUID
		}
		
		msg = Message{
			Type: MessageTypeCallEnded,
			Data: CallEvent{
				UUID:     uuid,
				State:    "ended",
				Duration: parseIntHeader(event.Headers["Variable_CDR(billsec)"]),
			},
		}
		
	case "UserEvent":
		// Handle custom AirwavePBX events
		if event.Headers["UserEvent"] != "" && event.Headers["Action"] == "UserEvent" {
			msg = Message{
				Type: MessageTypeEvent,
				Data: map[string]interface{}{
					"name":    event.Headers["UserEvent"],
					"headers": event.Headers,
				},
			}
		}
		
	default:
		// Don't broadcast all events, only relevant ones
		return
	}
	
	// Only broadcast if we have a valid message
	if msg.Type != "" {
		data, err := json.Marshal(msg)
		if err != nil {
			log.Printf("Failed to marshal event: %v", err)
			return
		}
		
		h.broadcast <- data
	}
}

// BroadcastMessage sends a message to all clients
func (h *Hub) BroadcastMessage(msgType string, data interface{}) {
	msg := Message{
		Type: msgType,
		Data: data,
	}
	
	jsonData, err := json.Marshal(msg)
	if err != nil {
		log.Printf("Failed to marshal message: %v", err)
		return
	}
	
	h.broadcast <- jsonData
}

// sendStats sends current system stats to all clients
func (h *Hub) sendStats() {
	stats := map[string]interface{}{
		"connected_clients": len(h.clients),
		"active_calls":      0, // TODO: Get from call service
		"system_status":     "online",
	}
	
	h.BroadcastMessage(MessageTypeStats, stats)
}

// parseIntHeader safely parses an integer from header
func parseIntHeader(value string) int {
	var result int
	if value != "" {
		_, _ = fmt.Sscanf(value, "%d", &result)
	}
	return result
}

// Handler returns the WebSocket handler function
func Handler(hub *Hub) func(*fiber.Ctx) error {
	return websocket.New(func(c *websocket.Conn) {
		client := &Client{
			conn: c,
			send: make(chan []byte, 256),
			hub:  hub,
			id:   c.RemoteAddr().String(),
		}
		
		hub.register <- client
		
		// Start goroutines for reading and writing
		go client.writePump()
		client.readPump()
		
		hub.unregister <- client
	})
}

// readPump handles incoming messages from the WebSocket
func (c *Client) readPump() {
	defer c.conn.Close()
	
	for {
		messageType, message, err := c.conn.ReadMessage()
		if err != nil {
			if websocket.IsUnexpectedCloseError(err, websocket.CloseGoingAway, websocket.CloseAbnormalClosure) {
				log.Printf("WebSocket error: %v", err)
			}
			break
		}
		
		// Handle ping messages
		if messageType == websocket.TextMessage {
			var msg Message
			if err := json.Unmarshal(message, &msg); err == nil && msg.Type == MessageTypePing {
				pong := Message{Type: MessageTypePong}
				if data, err := json.Marshal(pong); err == nil {
					c.send <- data
				}
			}
		}
	}
}

// writePump handles sending messages to the WebSocket
func (c *Client) writePump() {
	defer c.conn.Close()
	
	for {
		select {
		case message, ok := <-c.send:
			if !ok {
				// Hub closed the channel
				c.conn.WriteMessage(websocket.CloseMessage, []byte{})
				return
			}
			
			if err := c.conn.WriteMessage(websocket.TextMessage, message); err != nil {
				return
			}
		}
	}
}