// Dashboard Page

async function loadDashboardPage() {
    const contentDiv = document.getElementById('page-content');
    
    // Get system status and stats
    const [status, activeCalls, extensions, trunks] = await Promise.all([
        API.getSystemStatus(),
        API.getActiveCalls(),
        API.getExtensions(),
        API.getTrunks()
    ]);
    
    // Calculate stats
    const activeCallsCount = activeCalls.length;
    const extensionsCount = extensions.length;
    const onlineExtensions = extensions.filter(e => e.enabled).length;
    const trunksCount = trunks.length;
    const registeredTrunks = trunks.filter(t => t.enabled).length;
    
    contentDiv.innerHTML = `
        <div class="page-header">
            <h1 class="page-title">Dashboard</h1>
            <div class="page-actions">
                <button class="btn btn-primary" onclick="refreshDashboard()">
                    <i class="fap fa-sync"></i>
                    <span>Refresh</span>
                </button>
            </div>
        </div>
        
        <div class="stats-grid">
            <div class="stat-card glass-card">
                <i class="fap fa-phone-alt stat-icon text-primary"></i>
                <div class="stat-value">${activeCallsCount}</div>
                <div class="stat-label">Active Calls</div>
            </div>
            
            <div class="stat-card glass-card">
                <i class="fap fa-users stat-icon text-success"></i>
                <div class="stat-value">${onlineExtensions}/${extensionsCount}</div>
                <div class="stat-label">Extensions Online</div>
            </div>
            
            <div class="stat-card glass-card">
                <i class="fap fa-network-wired stat-icon text-info"></i>
                <div class="stat-value">${registeredTrunks}/${trunksCount}</div>
                <div class="stat-label">Trunks Registered</div>
            </div>
            
            <div class="stat-card glass-card">
                <i class="fap fa-server stat-icon text-success"></i>
                <div class="stat-value">${Math.floor(status.uptime / 3600)}h</div>
                <div class="stat-label">System Uptime</div>
            </div>
        </div>
        
        <div class="dashboard-grid">
            <div class="dashboard-section">
                <h2>Recent Calls</h2>
                <div id="recent-calls-list" class="glass-card">
                    ${await renderRecentCalls()}
                </div>
            </div>
            
            <div class="dashboard-section">
                <h2>System Health</h2>
                <div class="glass-card">
                    ${renderSystemHealth(status)}
                </div>
            </div>
        </div>
    `;
}

async function renderRecentCalls() {
    try {
        const { calls } = await API.getCallHistory(10, 0);
        
        if (calls.length === 0) {
            return '<p class="text-center text-secondary">No recent calls</p>';
        }
        
        return `
            <table class="glass-table">
                <thead>
                    <tr>
                        <th>Time</th>
                        <th>Caller</th>
                        <th>Duration</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    ${calls.map(call => `
                        <tr>
                            <td>${formatDateTime(call.start_time)}</td>
                            <td>
                                <div>${call.caller_id_name || 'Unknown'}</div>
                                <small class="text-secondary">${formatPhoneNumber(call.caller_id_number)}</small>
                            </td>
                            <td>${formatDuration(call.duration)}</td>
                            <td>
                                <span class="badge ${getCallStatusClass(call.disposition)}">
                                    ${call.disposition}
                                </span>
                            </td>
                        </tr>
                    `).join('')}
                </tbody>
            </table>
        `;
    } catch (error) {
        return '<p class="text-danger">Failed to load recent calls</p>';
    }
}

function renderSystemHealth(status) {
    const services = status.services || {};
    
    return `
        <div class="health-grid">
            ${Object.entries(services).map(([service, state]) => `
                <div class="health-item">
                    <div class="health-indicator ${state === 'running' ? 'success' : 'danger'}"></div>
                    <div class="health-label">${service.charAt(0).toUpperCase() + service.slice(1)}</div>
                    <div class="health-status">${state}</div>
                </div>
            `).join('')}
        </div>
        
        <div class="system-info mt-4">
            <div class="info-row">
                <span>Version:</span>
                <span>${status.version}</span>
            </div>
            <div class="info-row">
                <span>Uptime:</span>
                <span>${formatUptime(status.uptime)}</span>
            </div>
        </div>
    `;
}

function getCallStatusClass(disposition) {
    switch (disposition) {
        case 'ANSWERED':
            return 'success';
        case 'BUSY':
            return 'warning';
        case 'NO ANSWER':
        case 'FAILED':
            return 'danger';
        default:
            return 'secondary';
    }
}

function formatUptime(seconds) {
    const days = Math.floor(seconds / 86400);
    const hours = Math.floor((seconds % 86400) / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    
    const parts = [];
    if (days > 0) parts.push(`${days}d`);
    if (hours > 0) parts.push(`${hours}h`);
    if (minutes > 0) parts.push(`${minutes}m`);
    
    return parts.join(' ') || '0m';
}

async function refreshDashboard() {
    showNotification('Refreshing', 'Updating dashboard data...', 'info');
    await loadDashboardPage();
}

// Add dashboard-specific styles
const dashboardStyles = `
<style>
.dashboard-grid {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: var(--spacing-lg);
    margin-top: var(--spacing-xl);
}

.dashboard-section h2 {
    font-size: 20px;
    margin-bottom: var(--spacing-md);
}

.health-grid {
    display: grid;
    gap: var(--spacing-md);
}

.health-item {
    display: flex;
    align-items: center;
    gap: var(--spacing-md);
}

.health-indicator {
    width: 12px;
    height: 12px;
    border-radius: 50%;
    flex-shrink: 0;
}

.health-indicator.success {
    background: var(--success-color);
    box-shadow: 0 0 8px var(--success-color);
}

.health-indicator.danger {
    background: var(--danger-color);
    box-shadow: 0 0 8px var(--danger-color);
}

.health-label {
    flex: 1;
    font-weight: 500;
}

.health-status {
    color: var(--text-secondary);
    font-size: 14px;
}

.system-info {
    padding-top: var(--spacing-md);
    border-top: 1px solid var(--glass-border);
}

.info-row {
    display: flex;
    justify-content: space-between;
    padding: var(--spacing-xs) 0;
    font-size: 14px;
}

.info-row span:first-child {
    color: var(--text-secondary);
}

.badge {
    padding: 2px 8px;
    border-radius: 4px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
}

.badge.success {
    background: rgba(52, 199, 89, 0.2);
    color: var(--success-color);
}

.badge.warning {
    background: rgba(255, 149, 0, 0.2);
    color: var(--warning-color);
}

.badge.danger {
    background: rgba(255, 59, 48, 0.2);
    color: var(--danger-color);
}

.badge.secondary {
    background: var(--glass-bg);
    color: var(--text-secondary);
}

@media (max-width: 1024px) {
    .dashboard-grid {
        grid-template-columns: 1fr;
    }
}
</style>
`;

// Inject styles
if (!document.getElementById('dashboard-styles')) {
    const styleElement = document.createElement('div');
    styleElement.id = 'dashboard-styles';
    styleElement.innerHTML = dashboardStyles;
    document.head.appendChild(styleElement);
}