package auth

import (
	"crypto/subtle"
	"errors"
	"time"
	
	"github.com/golang-jwt/jwt/v5"
	"golang.org/x/crypto/bcrypt"
)

var (
	ErrInvalidCredentials = errors.New("invalid credentials")
	ErrInvalidToken       = errors.New("invalid token")
	ErrTokenExpired       = errors.New("token expired")
)

// Service handles authentication
type Service struct {
	adminUsername string
	adminPassword string
	apiKey        string
	jwtSecret     []byte
}

// Claims represents JWT claims
type Claims struct {
	Username string `json:"username"`
	Admin    bool   `json:"admin"`
	jwt.RegisteredClaims
}

// NewService creates a new auth service
func NewService(username, password, apiKey string) *Service {
	// Hash the admin password
	hashedPassword, _ := bcrypt.GenerateFromPassword([]byte(password), bcrypt.DefaultCost)
	
	return &Service{
		adminUsername: username,
		adminPassword: string(hashedPassword),
		apiKey:        apiKey,
		jwtSecret:     []byte(apiKey), // Use API key as JWT secret
	}
}

// ValidateCredentials checks username and password
func (s *Service) ValidateCredentials(username, password string) error {
	if username != s.adminUsername {
		return ErrInvalidCredentials
	}
	
	if err := bcrypt.CompareHashAndPassword([]byte(s.adminPassword), []byte(password)); err != nil {
		return ErrInvalidCredentials
	}
	
	return nil
}

// ValidateAPIKey checks if the API key is valid
func (s *Service) ValidateAPIKey(key string) bool {
	return subtle.ConstantTimeCompare([]byte(key), []byte(s.apiKey)) == 1
}

// GenerateToken creates a JWT token
func (s *Service) GenerateToken(username string) (string, error) {
	claims := &Claims{
		Username: username,
		Admin:    username == s.adminUsername,
		RegisteredClaims: jwt.RegisteredClaims{
			ExpiresAt: jwt.NewNumericDate(time.Now().Add(24 * time.Hour)),
			IssuedAt:  jwt.NewNumericDate(time.Now()),
			NotBefore: jwt.NewNumericDate(time.Now()),
			Issuer:    "AirwavePBX",
		},
	}
	
	token := jwt.NewWithClaims(jwt.SigningMethodHS256, claims)
	return token.SignedString(s.jwtSecret)
}

// ValidateToken verifies and parses a JWT token
func (s *Service) ValidateToken(tokenString string) (*Claims, error) {
	token, err := jwt.ParseWithClaims(tokenString, &Claims{}, func(token *jwt.Token) (interface{}, error) {
		if _, ok := token.Method.(*jwt.SigningMethodHMAC); !ok {
			return nil, ErrInvalidToken
		}
		return s.jwtSecret, nil
	})
	
	if err != nil {
		return nil, err
	}
	
	if claims, ok := token.Claims.(*Claims); ok && token.Valid {
		return claims, nil
	}
	
	return nil, ErrInvalidToken
}