package services

import (
	"fmt"
	
	"io/ioutil"
	"os"
	"strings"
	
	"airwavepbx/internal/ami"
	"airwavepbx/internal/database"
	"golang.org/x/crypto/bcrypt"
)

// Extension represents a SIP extension
type Extension struct {
	ID                int    `json:"id"`
	Extension         string `json:"extension"`
	Name              string `json:"name"`
	Password          string `json:"password,omitempty"`
	Context           string `json:"context"`
	CallerIDName      string `json:"callerid_name"`
	CallerIDNumber    string `json:"callerid_number"`
	VoicemailEnabled  bool   `json:"voicemail_enabled"`
	VoicemailPassword string `json:"voicemail_password,omitempty"`
	Enabled           bool   `json:"enabled"`
}

// ExtensionService handles extension-related operations
type ExtensionService struct {
	db  *database.DB
	ami *ami.Client
}

// NewExtensionService creates a new extension service
func NewExtensionService(db *database.DB, ami *ami.Client) *ExtensionService {
	return &ExtensionService{
		db:  db,
		ami: ami,
	}
}

// GetAll returns all extensions
func (s *ExtensionService) GetAll() ([]Extension, error) {
	query := `
		SELECT id, extension, name, context, callerid_name, callerid_number,
		       voicemail_enabled, enabled
		FROM extensions
		ORDER BY extension
	`
	
	rows, err := s.db.Query(query)
	if err != nil {
		return nil, err
	}
	defer rows.Close()
	
	var extensions []Extension
	for rows.Next() {
		var ext Extension
		err := rows.Scan(
			&ext.ID, &ext.Extension, &ext.Name, &ext.Context,
			&ext.CallerIDName, &ext.CallerIDNumber,
			&ext.VoicemailEnabled, &ext.Enabled,
		)
		if err != nil {
			return nil, err
		}
		extensions = append(extensions, ext)
	}
	
	return extensions, nil
}

// Get returns a specific extension
func (s *ExtensionService) Get(id int) (*Extension, error) {
	var ext Extension
	query := `
		SELECT id, extension, name, context, callerid_name, callerid_number,
		       voicemail_enabled, voicemail_password, enabled
		FROM extensions
		WHERE id = ?
	`
	
	err := s.db.QueryRow(query, id).Scan(
		&ext.ID, &ext.Extension, &ext.Name, &ext.Context,
		&ext.CallerIDName, &ext.CallerIDNumber,
		&ext.VoicemailEnabled, &ext.VoicemailPassword, &ext.Enabled,
	)
	
	if err != nil {
		return nil, err
	}
	
	return &ext, nil
}

// Create adds a new extension
func (s *ExtensionService) Create(ext *Extension) error {
	// Hash the password
	hashedPassword, err := bcrypt.GenerateFromPassword([]byte(ext.Password), bcrypt.DefaultCost)
	if err != nil {
		return err
	}
	
	// Set defaults
	if ext.Context == "" {
		ext.Context = "default"
	}
	if ext.CallerIDName == "" {
		ext.CallerIDName = ext.Name
	}
	if ext.CallerIDNumber == "" {
		ext.CallerIDNumber = ext.Extension
	}
	
	// Insert into database
	result, err := s.db.Exec(`
		INSERT INTO extensions (extension, name, password, context, 
		                       callerid_name, callerid_number, 
		                       voicemail_enabled, voicemail_password, enabled)
		VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
	`, ext.Extension, ext.Name, string(hashedPassword), ext.Context,
		ext.CallerIDName, ext.CallerIDNumber,
		ext.VoicemailEnabled, ext.VoicemailPassword, ext.Enabled)
	
	if err != nil {
		return err
	}
	
	id, _ := result.LastInsertId()
	ext.ID = int(id)
	
	// Update Asterisk configuration
	return s.updateAsteriskConfig()
}

// Update modifies an existing extension
func (s *ExtensionService) Update(ext *Extension) error {
	// If password is provided, hash it
	if ext.Password != "" {
		hashedPassword, err := bcrypt.GenerateFromPassword([]byte(ext.Password), bcrypt.DefaultCost)
		if err != nil {
			return err
		}
		
		_, err = s.db.Exec(`
			UPDATE extensions 
			SET name = ?, password = ?, context = ?, 
			    callerid_name = ?, callerid_number = ?,
			    voicemail_enabled = ?, voicemail_password = ?, enabled = ?
			WHERE id = ?
		`, ext.Name, string(hashedPassword), ext.Context,
			ext.CallerIDName, ext.CallerIDNumber,
			ext.VoicemailEnabled, ext.VoicemailPassword, ext.Enabled, ext.ID)
		
		if err != nil {
			return err
		}
	} else {
		// Update without changing password
		_, err := s.db.Exec(`
			UPDATE extensions 
			SET name = ?, context = ?, 
			    callerid_name = ?, callerid_number = ?,
			    voicemail_enabled = ?, voicemail_password = ?, enabled = ?
			WHERE id = ?
		`, ext.Name, ext.Context,
			ext.CallerIDName, ext.CallerIDNumber,
			ext.VoicemailEnabled, ext.VoicemailPassword, ext.Enabled, ext.ID)
		
		if err != nil {
			return err
		}
	}
	
	// Update Asterisk configuration
	return s.updateAsteriskConfig()
}

// Delete removes an extension
func (s *ExtensionService) Delete(id int) error {
	_, err := s.db.Exec("DELETE FROM extensions WHERE id = ?", id)
	if err != nil {
		return err
	}
	
	// Update Asterisk configuration
	return s.updateAsteriskConfig()
}

// updateAsteriskConfig updates the Asterisk PJSIP configuration
func (s *ExtensionService) updateAsteriskConfig() error {
	// Generate PJSIP configuration for all extensions
	extensions, err := s.GetAll()
	if err != nil {
		return err
	}
	
	// Build PJSIP configuration
	config := "; AirwavePBX Generated PJSIP Extensions\n\n"
	
	for _, ext := range extensions {
		if !ext.Enabled {
			continue
		}
		
		// Create endpoint configuration
		config += fmt.Sprintf(`; Extension %s - %s
[%s]
type=endpoint
context=%s
disallow=all
allow=opus
allow=g722
allow=ulaw
allow=alaw
auth=%s
aors=%s
callerid="%s" <%s>
use_avpf=yes
media_encryption=sdes
dtls_verify=fingerprint
dtls_setup=actpass
ice_support=yes
media_use_received_transport=yes
rtcp_mux=yes

[%s]
type=auth
auth_type=userpass
username=%s
password=%s

[%s]
type=aor
max_contacts=5
qualify_frequency=60
remove_existing=yes

`,
			ext.Extension, ext.Name,
			ext.Extension, ext.Context, ext.Extension, ext.Extension,
			ext.CallerIDName, ext.CallerIDNumber,
			ext.Extension, ext.Extension, "PLACEHOLDER_PASSWORD", // Password will be handled separately
			ext.Extension)
	}
	
	// Write to Asterisk PJSIP configuration directory
	configPath := "/etc/asterisk/pjsip_airwave_extensions.conf"
	err = ioutil.WriteFile(configPath, []byte(config), 0644)
	if err != nil {
		return fmt.Errorf("failed to write PJSIP config: %w", err)
	}
	
	// Ensure the file is included in the main pjsip.conf
	mainConfig := "/etc/asterisk/pjsip.conf"
	includeDirective := "#include pjsip_airwave_extensions.conf\n"
	
	// Check if include already exists
	content, err := ioutil.ReadFile(mainConfig)
	if err == nil && !strings.Contains(string(content), includeDirective) {
		// Append include directive
		f, err := os.OpenFile(mainConfig, os.O_APPEND|os.O_WRONLY, 0644)
		if err != nil {
			return fmt.Errorf("failed to update main pjsip.conf: %w", err)
		}
		defer f.Close()
		
		if _, err := f.WriteString("\n" + includeDirective); err != nil {
			return fmt.Errorf("failed to write include directive: %w", err)
		}
	}
	
	// Reload PJSIP module in Asterisk
	action := ami.Action{
		Name: "Command",
		Headers: map[string]string{
			"Command": "pjsip reload",
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return fmt.Errorf("failed to reload PJSIP: %w", err)
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to reload PJSIP config: %s", response.Headers["Message"])
	}
	
	return nil
}