// API Documentation Page

async function loadAPIDocsPage() {
    const contentDiv = document.getElementById('page-content');
    
    try {
        const docs = await API.getAPIDocs();
        
        contentDiv.innerHTML = `
            <div class="page-header">
                <h1 class="page-title">API Documentation</h1>
                <div class="page-actions">
                    <button class="btn" onclick="copyAPIKey()">
                        <i class="fap fa-key"></i>
                        <span>Copy API Key</span>
                    </button>
                </div>
            </div>
            
            <div class="docs-container">
                <div class="docs-sidebar glass-card">
                    <h3>Quick Navigation</h3>
                    <ul class="docs-nav">
                        <li><a href="#overview" onclick="scrollToSection('overview')">Overview</a></li>
                        <li><a href="#authentication" onclick="scrollToSection('authentication')">Authentication</a></li>
                        <li><a href="#endpoints" onclick="scrollToSection('endpoints')">REST Endpoints</a></li>
                        <li><a href="#websocket" onclick="scrollToSection('websocket')">WebSocket Events</a></li>
                        <li><a href="#examples" onclick="scrollToSection('examples')">Code Examples</a></li>
                    </ul>
                </div>
                
                <div class="docs-content">
                    <section id="overview" class="docs-section glass-card">
                        <h2>Overview</h2>
                        <p>AirwavePBX provides a comprehensive REST API and WebSocket interface for integrating with broadcasting applications and custom telephony solutions.</p>
                        
                        <div class="info-box">
                            <strong>Base URL:</strong> <code>${docs.base_url}</code><br>
                            <strong>Version:</strong> ${docs.version}<br>
                            <strong>Format:</strong> JSON
                        </div>
                    </section>
                    
                    <section id="authentication" class="docs-section glass-card">
                        <h2>Authentication</h2>
                        <p>The API supports two authentication methods:</p>
                        
                        <h3>1. Bearer Token (JWT)</h3>
                        <div class="code-block">
Authorization: Bearer &lt;token&gt;
                        </div>
                        
                        <h3>2. API Key</h3>
                        <div class="code-block">
X-API-Key: &lt;your-api-key&gt;
                        </div>
                        
                        <p>API keys are recommended for server-to-server communication and never expire.</p>
                    </section>
                    
                    <section id="endpoints" class="docs-section glass-card">
                        <h2>REST Endpoints</h2>
                        ${renderEndpoints(docs.endpoints)}
                    </section>
                    
                    <section id="websocket" class="docs-section glass-card">
                        <h2>WebSocket Events</h2>
                        <p>Connect to receive real-time call events:</p>
                        
                        <div class="info-box">
                            <strong>URL:</strong> <code>${docs.websocket.url}</code><br>
                            <strong>Authentication:</strong> ${docs.websocket.authentication}
                        </div>
                        
                        <h3>Available Events</h3>
                        ${renderWebSocketEvents(docs.websocket.events)}
                    </section>
                    
                    <section id="examples" class="docs-section glass-card">
                        <h2>Code Examples</h2>
                        ${renderCodeExamples(docs.examples)}
                    </section>
                </div>
            </div>
        `;
        
        // Initialize code highlighting if available
        if (window.Prism) {
            Prism.highlightAll();
        }
        
    } catch (error) {
        contentDiv.innerHTML = '<div class="error">Failed to load API documentation</div>';
    }
}

function renderEndpoints(endpoints) {
    let html = '';
    
    for (const [category, categoryEndpoints] of Object.entries(endpoints)) {
        html += `<h3>${category.charAt(0).toUpperCase() + category.slice(1)}</h3>`;
        
        for (const [endpoint, details] of Object.entries(categoryEndpoints)) {
            const [method, path] = endpoint.split(' ');
            
            html += `
                <div class="api-endpoint glass-card">
                    <div class="endpoint-header">
                        <span class="method-badge ${method.toLowerCase()}">${method}</span>
                        <code class="endpoint-path">${path}</code>
                    </div>
                    <p>${details.description}</p>
                    
                    ${details.auth ? '<p class="text-warning"><i class="fap fa-lock"></i> Authentication required</p>' : ''}
                    
                    ${details.body ? `
                        <h4>Request Body</h4>
                        <div class="code-block">
${JSON.stringify(details.body, null, 2)}
                        </div>
                    ` : ''}
                    
                    ${details.query ? `
                        <h4>Query Parameters</h4>
                        <div class="code-block">
${JSON.stringify(details.query, null, 2)}
                        </div>
                    ` : ''}
                    
                    ${details.response ? `
                        <h4>Response</h4>
                        <p>${details.response}</p>
                    ` : ''}
                </div>
            `;
        }
    }
    
    return html;
}

function renderWebSocketEvents(events) {
    let html = '<div class="events-grid">';
    
    for (const [event, details] of Object.entries(events)) {
        html += `
            <div class="event-card glass-card">
                <h4><i class="fap fa-bolt"></i> ${event}</h4>
                <p>${details.description}</p>
                <div class="code-block">
${JSON.stringify(details.data, null, 2)}
                </div>
            </div>
        `;
    }
    
    html += '</div>';
    return html;
}

function renderCodeExamples(examples) {
    let html = '<div class="examples-tabs">';
    
    // Tab buttons
    html += '<div class="tab-buttons">';
    for (const language of Object.keys(examples)) {
        html += `
            <button class="tab-button ${language === 'javascript' ? 'active' : ''}" 
                    onclick="showExampleTab('${language}')">
                ${language.charAt(0).toUpperCase() + language.slice(1)}
            </button>
        `;
    }
    html += '</div>';
    
    // Tab content
    for (const [language, languageExamples] of Object.entries(examples)) {
        html += `
            <div class="tab-content ${language}" style="${language !== 'javascript' ? 'display: none;' : ''}">
        `;
        
        for (const [title, code] of Object.entries(languageExamples)) {
            html += `
                <h4>${title.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}</h4>
                <div class="code-block language-${language}">
<pre><code>${escapeHtml(code)}</code></pre>
                </div>
            `;
        }
        
        html += '</div>';
    }
    
    html += '</div>';
    return html;
}

function showExampleTab(language) {
    // Hide all tabs
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.style.display = 'none';
    });
    
    // Remove active class from all buttons
    document.querySelectorAll('.tab-button').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Show selected tab
    document.querySelector(`.tab-content.${language}`).style.display = 'block';
    event.target.classList.add('active');
}

function scrollToSection(section) {
    document.getElementById(section).scrollIntoView({ behavior: 'smooth' });
}

async function copyAPIKey() {
    // In production, this would get the actual API key
    const apiKey = 'your-api-key-here';
    
    try {
        await navigator.clipboard.writeText(apiKey);
        showNotification('Success', 'API key copied to clipboard', 'success');
    } catch (error) {
        showNotification('Error', 'Failed to copy API key', 'error');
    }
}

function escapeHtml(unsafe) {
    return unsafe
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
}

// Add API docs-specific styles
const apiDocsStyles = `
<style>
.docs-container {
    display: grid;
    grid-template-columns: 250px 1fr;
    gap: var(--spacing-lg);
    align-items: start;
}

.docs-sidebar {
    position: sticky;
    top: var(--spacing-lg);
    padding: var(--spacing-lg);
}

.docs-sidebar h3 {
    font-size: 16px;
    margin-bottom: var(--spacing-md);
}

.docs-nav {
    list-style: none;
    padding: 0;
}

.docs-nav li {
    margin-bottom: var(--spacing-sm);
}

.docs-nav a {
    color: var(--text-secondary);
    text-decoration: none;
    transition: color var(--transition-fast);
}

.docs-nav a:hover {
    color: var(--text-primary);
}

.docs-content {
    display: flex;
    flex-direction: column;
    gap: var(--spacing-lg);
}

.docs-section {
    padding: var(--spacing-xl);
}

.docs-section h2 {
    font-size: 28px;
    margin-bottom: var(--spacing-md);
}

.docs-section h3 {
    font-size: 20px;
    margin-top: var(--spacing-lg);
    margin-bottom: var(--spacing-md);
}

.docs-section h4 {
    font-size: 16px;
    margin-top: var(--spacing-md);
    margin-bottom: var(--spacing-sm);
}

.info-box {
    background: var(--glass-bg);
    border: 1px solid var(--glass-border);
    border-radius: var(--radius-md);
    padding: var(--spacing-md);
    margin: var(--spacing-md) 0;
    font-family: monospace;
    font-size: 14px;
}

.endpoint-path {
    font-size: 16px;
    font-weight: 500;
}

.events-grid {
    display: grid;
    gap: var(--spacing-md);
}

.event-card {
    padding: var(--spacing-md);
    background: var(--glass-bg);
}

.event-card h4 {
    display: flex;
    align-items: center;
    gap: var(--spacing-sm);
    margin-bottom: var(--spacing-sm);
}

.examples-tabs {
    margin-top: var(--spacing-lg);
}

.tab-buttons {
    display: flex;
    gap: var(--spacing-sm);
    margin-bottom: var(--spacing-lg);
    border-bottom: 1px solid var(--glass-border);
}

.tab-button {
    background: transparent;
    border: none;
    color: var(--text-secondary);
    padding: var(--spacing-sm) var(--spacing-md);
    cursor: pointer;
    position: relative;
    transition: color var(--transition-fast);
}

.tab-button:hover {
    color: var(--text-primary);
}

.tab-button.active {
    color: var(--primary-color);
}

.tab-button.active::after {
    content: '';
    position: absolute;
    bottom: -1px;
    left: 0;
    right: 0;
    height: 2px;
    background: var(--primary-color);
}

.code-block pre {
    margin: 0;
    white-space: pre-wrap;
    word-wrap: break-word;
}

.code-block code {
    color: var(--text-primary);
}

@media (max-width: 1024px) {
    .docs-container {
        grid-template-columns: 1fr;
    }
    
    .docs-sidebar {
        position: static;
    }
}
</style>
`;

if (!document.getElementById('api-docs-styles')) {
    const styleElement = document.createElement('div');
    styleElement.id = 'api-docs-styles';
    styleElement.innerHTML = apiDocsStyles;
    document.head.appendChild(styleElement);
}