// Trunks Page

async function loadTrunksPage() {
    const contentDiv = document.getElementById('page-content');
    
    contentDiv.innerHTML = `
        <div class="page-header">
            <h1 class="page-title">Trunks</h1>
            <div class="page-actions">
                <button class="btn btn-primary" onclick="showAddTrunk()">
                    <i class="fap fa-plus"></i>
                    <span>Add Trunk</span>
                </button>
            </div>
        </div>
        
        <div class="glass-card">
            <div id="trunks-table-container">
                <div class="loading">Loading trunks...</div>
            </div>
        </div>
    `;
    
    await loadTrunks();
}

async function loadTrunks() {
    try {
        const trunks = await API.getTrunks();
        renderTrunksTable(trunks);
    } catch (error) {
        showNotification('Error', 'Failed to load trunks', 'error');
    }
}

function renderTrunksTable(trunks) {
    const container = document.getElementById('trunks-table-container');
    
    if (trunks.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <i class="fap fa-network-wired" style="font-size: 48px; color: var(--text-tertiary);"></i>
                <p class="mt-3">No trunks configured</p>
                <p class="text-secondary">Add a trunk to connect to your VoIP provider</p>
                <button class="btn btn-primary mt-3" onclick="showAddTrunk()">
                    <i class="fap fa-plus"></i>
                    <span>Add First Trunk</span>
                </button>
            </div>
        `;
        return;
    }
    
    container.innerHTML = `
        <table class="glass-table">
            <thead>
                <tr>
                    <th>Name</th>
                    <th>Provider</th>
                    <th>Host</th>
                    <th>Username</th>
                    <th>Registration</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                ${trunks.map(trunk => `
                    <tr>
                        <td>
                            <span class="trunk-name">${trunk.name}</span>
                        </td>
                        <td>
                            <span class="provider-badge ${trunk.provider.toLowerCase()}">
                                ${getProviderIcon(trunk.provider)}
                                ${trunk.provider}
                            </span>
                        </td>
                        <td>${trunk.host}</td>
                        <td>${trunk.username}</td>
                        <td>
                            <span class="badge ${trunk.register ? 'success' : 'secondary'}">
                                ${trunk.register ? 'Yes' : 'No'}
                            </span>
                        </td>
                        <td>
                            <div class="status-indicator ${trunk.enabled ? 'success' : 'danger'}"></div>
                            ${trunk.enabled ? 'Active' : 'Inactive'}
                        </td>
                        <td>
                            <div class="action-buttons">
                                <button class="btn-icon" onclick="testTrunk(${trunk.id})" title="Test Connection">
                                    <i class="fap fa-plug"></i>
                                </button>
                                <button class="btn-icon" onclick="editTrunk(${trunk.id})" title="Edit">
                                    <i class="fap fa-edit"></i>
                                </button>
                                <button class="btn-icon" onclick="confirmDeleteTrunk(${trunk.id}, '${trunk.name}')" title="Delete">
                                    <i class="fap fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                `).join('')}
            </tbody>
        </table>
    `;
}

function getProviderIcon(provider) {
    const icons = {
        'callcentric': 'fa-phone-square',
        'voip.ms': 'fa-phone-office',
        'voipms': 'fa-phone-office',
        'twilio': 'fa-code-branch',
        'flowroute': 'fa-route'
    };
    
    const icon = icons[provider.toLowerCase()] || 'fa-server';
    return `<i class="fap ${icon}"></i>`;
}

function showAddTrunk() {
    showTrunkForm('Add Trunk', null);
}

async function editTrunk(id) {
    try {
        const trunk = await API.getTrunk(id);
        showTrunkForm('Edit Trunk', trunk);
    } catch (error) {
        showNotification('Error', 'Failed to load trunk details', 'error');
    }
}

function showTrunkForm(title, trunk) {
    const isEdit = !!trunk;
    
    showModal(title, `
        <form id="trunk-form" onsubmit="handleTrunkSubmit(event, ${isEdit ? trunk.id : 'null'})">
            <div class="form-group">
                <label>Trunk Name *</label>
                <div class="glass-input">
                    <i class="fap fa-tag input-icon"></i>
                    <input type="text" name="name" value="${trunk?.name || ''}" 
                           placeholder="e.g., Main Trunk" required>
                </div>
            </div>
            
            <div class="form-group">
                <label>Provider *</label>
                <select name="provider" class="glass-input" onchange="updateProviderDefaults(this.value)" required>
                    <option value="">Select Provider</option>
                    <option value="callcentric" ${trunk?.provider === 'callcentric' ? 'selected' : ''}>Callcentric</option>
                    <option value="voip.ms" ${trunk?.provider === 'voip.ms' ? 'selected' : ''}>VoIP.ms</option>
                    <option value="twilio" ${trunk?.provider === 'twilio' ? 'selected' : ''}>Twilio</option>
                    <option value="flowroute" ${trunk?.provider === 'flowroute' ? 'selected' : ''}>Flowroute</option>
                    <option value="other" ${trunk?.provider === 'other' ? 'selected' : ''}>Other</option>
                </select>
            </div>
            
            <div class="form-group">
                <label>Host/Server *</label>
                <div class="glass-input">
                    <i class="fap fa-server input-icon"></i>
                    <input type="text" name="host" value="${trunk?.host || ''}" 
                           placeholder="e.g., sip.provider.com" required>
                </div>
            </div>
            
            <div class="form-group">
                <label>Username *</label>
                <div class="glass-input">
                    <i class="fap fa-user input-icon"></i>
                    <input type="text" name="username" value="${trunk?.username || ''}" 
                           placeholder="Your account username" required>
                </div>
            </div>
            
            <div class="form-group">
                <label>Password *</label>
                <div class="glass-input">
                    <i class="fap fa-lock input-icon"></i>
                    <input type="password" name="password" value="${trunk?.password || ''}" 
                           placeholder="Your account password" required>
                </div>
            </div>
            
            <div class="form-group">
                <label>From User</label>
                <div class="glass-input">
                    <i class="fap fa-user-tag input-icon"></i>
                    <input type="text" name="from_user" value="${trunk?.from_user || ''}" 
                           placeholder="Usually same as username">
                </div>
            </div>
            
            <div class="form-group">
                <label>From Domain</label>
                <div class="glass-input">
                    <i class="fap fa-globe input-icon"></i>
                    <input type="text" name="from_domain" value="${trunk?.from_domain || ''}" 
                           placeholder="Usually same as host">
                </div>
            </div>
            
            <div class="form-group">
                <label>Codec Preferences</label>
                <div class="glass-input">
                    <i class="fap fa-microphone input-icon"></i>
                    <input type="text" name="codec_prefs" value="${trunk?.codec_prefs || 'OPUS,G722,PCMU,PCMA'}" 
                           placeholder="OPUS,G722,PCMU,PCMA">
                </div>
            </div>
            
            <div class="form-group">
                <label class="checkbox-label">
                    <input type="checkbox" name="register" ${trunk?.register !== false ? 'checked' : ''}>
                    <span>Register with provider</span>
                </label>
            </div>
            
            <div class="form-group">
                <label class="checkbox-label">
                    <input type="checkbox" name="enabled" ${trunk?.enabled !== false ? 'checked' : ''}>
                    <span>Enable trunk</span>
                </label>
            </div>
        </form>
    `, [
        {
            label: 'Cancel',
            onclick: 'closeModal()'
        },
        {
            label: isEdit ? 'Update' : 'Create',
            class: 'btn-primary',
            icon: isEdit ? 'save' : 'plus',
            onclick: "document.getElementById('trunk-form').submit()"
        }
    ]);
}

function updateProviderDefaults(provider) {
    const form = document.getElementById('trunk-form');
    
    switch(provider) {
        case 'callcentric':
            form.host.value = 'callcentric.com';
            form.from_domain.value = 'callcentric.com';
            form.register.checked = true;
            break;
        case 'voip.ms':
            form.host.value = 'atlanta.voip.ms';
            form.from_domain.value = 'voip.ms';
            form.register.checked = true;
            break;
        case 'twilio':
            form.register.checked = false;
            form.from_domain.value = 'twilio.com';
            break;
        case 'flowroute':
            form.host.value = 'sip.flowroute.com';
            form.from_domain.value = 'flowroute.com';
            form.register.checked = false;
            break;
    }
}

async function handleTrunkSubmit(event, trunkId) {
    event.preventDefault();
    const formData = new FormData(event.target);
    
    const data = {
        name: formData.get('name'),
        provider: formData.get('provider'),
        host: formData.get('host'),
        username: formData.get('username'),
        password: formData.get('password'),
        from_user: formData.get('from_user') || formData.get('username'),
        from_domain: formData.get('from_domain') || formData.get('host'),
        codec_prefs: formData.get('codec_prefs'),
        register: formData.get('register') === 'on',
        enabled: formData.get('enabled') === 'on'
    };
    
    try {
        if (trunkId) {
            await API.updateTrunk(trunkId, data);
            showNotification('Success', 'Trunk updated successfully', 'success');
        } else {
            await API.createTrunk(data);
            showNotification('Success', 'Trunk created successfully', 'success');
        }
        closeModal();
        await loadTrunks();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

async function testTrunk(id) {
    const button = event.target.closest('button');
    const originalContent = button.innerHTML;
    button.innerHTML = '<i class="fap fa-spinner fa-spin"></i>';
    button.disabled = true;
    
    try {
        const result = await API.testTrunk(id);
        showNotification('Success', result.message || 'Trunk connection successful', 'success');
    } catch (error) {
        showNotification('Error', error.message || 'Trunk connection failed', 'error');
    } finally {
        button.innerHTML = originalContent;
        button.disabled = false;
    }
}

function confirmDeleteTrunk(id, name) {
    showModal('Delete Trunk', `
        <p>Are you sure you want to delete trunk <strong>${name}</strong>?</p>
        <p class="text-warning mt-3">
            <i class="fap fa-exclamation-triangle"></i>
            This will disconnect all calls using this trunk.
        </p>
    `, [
        {
            label: 'Cancel',
            onclick: 'closeModal()'
        },
        {
            label: 'Delete',
            class: 'btn-danger',
            icon: 'trash',
            onclick: `deleteTrunk(${id})`
        }
    ]);
}

async function deleteTrunk(id) {
    try {
        await API.deleteTrunk(id);
        closeModal();
        showNotification('Success', 'Trunk deleted successfully', 'success');
        await loadTrunks();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

// Add trunks-specific styles
const trunksStyles = `
<style>
.trunk-name {
    font-weight: 600;
    color: var(--primary-color);
}

.provider-badge {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 4px 10px;
    background: var(--glass-bg);
    border: 1px solid var(--glass-border);
    border-radius: var(--radius-sm);
    font-size: 13px;
    font-weight: 500;
}

.provider-badge.callcentric {
    background: rgba(0, 122, 255, 0.1);
    border-color: rgba(0, 122, 255, 0.3);
    color: var(--primary-color);
}

.provider-badge.voip\.ms,
.provider-badge.voipms {
    background: rgba(52, 199, 89, 0.1);
    border-color: rgba(52, 199, 89, 0.3);
    color: var(--success-color);
}

.provider-badge.twilio {
    background: rgba(255, 59, 48, 0.1);
    border-color: rgba(255, 59, 48, 0.3);
    color: var(--danger-color);
}

.provider-badge.flowroute {
    background: rgba(255, 149, 0, 0.1);
    border-color: rgba(255, 149, 0, 0.3);
    color: var(--warning-color);
}

select.glass-input {
    background: var(--glass-bg);
    border: 1px solid var(--glass-border);
    border-radius: var(--radius-md);
    padding: var(--spacing-sm) var(--spacing-md);
    color: var(--text-primary);
    width: 100%;
    outline: none;
    cursor: pointer;
}

select.glass-input:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 122, 255, 0.2);
}

select.glass-input option {
    background: var(--bg-secondary);
    color: var(--text-primary);
}

.fa-spinner.fa-spin {
    animation: fa-spin 1s linear infinite;
}
</style>
`;

if (!document.getElementById('trunks-styles')) {
    const styleElement = document.createElement('div');
    styleElement.id = 'trunks-styles';
    styleElement.innerHTML = trunksStyles;
    document.head.appendChild(styleElement);
}