#!/bin/bash
# AirwavePBX Professional Installation Script
# Version: 1.0.0 - FreeSWITCH-based architecture
# Target OS: Ubuntu 24.04 LTS

# Don't exit on errors - we'll handle them explicitly
set +e

# Set non-interactive mode for all package operations
export DEBIAN_FRONTEND=noninteractive
export APT_LISTCHANGES_FRONTEND=none

# Global error tracking
INSTALL_ERRORS=0

# Global configuration tracking
USE_STANDARD_PORTS=false
DOMAIN=""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
MAGENTA='\033[0;35m'
NC='\033[0m'
BOLD='\033[1m'

# Script version
INSTALLER_VERSION="1.0.2"
AIRWAVE_VERSION="1.0.2"

# Configuration
INSTALL_DIR="/opt/airwavepbx"
DATA_DIR="/var/lib/airwavepbx"
LOG_DIR="/var/log/airwavepbx"
CREDENTIALS_FILE="/root/airwavepbx-credentials.txt"
SYSTEMD_SERVICE="airwavepbx.service"
SSL_DIR="/etc/letsencrypt"
BACKUP_DIR="/var/backups/airwavepbx"

# Generate secure passwords
DB_PASSWORD=$(openssl rand -hex 32)
ADMIN_PASSWORD=$(openssl rand -base64 12 | tr -d "=+/" | cut -c1-16)
API_KEY=$(openssl rand -hex 32)
ESL_PASSWORD=$(openssl rand -hex 16)

# Print banner
print_banner() {
    clear
    echo -e "${CYAN}"
    cat << "EOF"
⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿
⣿⣿⣿⣿⣿⣿⣿⡏⠀⠀⢻⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿
⣿⣿⣿⣿⣿⣿⡟⠀⣰⡆⡘⠁⠹⣿⣿⣿⣿⣿⣿⣿⣿⠟⠛⣿⣿⡏⠉⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⡟⠛⠛⠛⠻⣿⡟⠛⠛⠛⠻⣟⠛⢻⣿⡟⠛⣿⣿⣿⣿
⣿⣿⣿⣿⣿⡿⠀⢰⣿⣿⣅⠀⢴⣿⣿⣿⣿⣿⣿⣿⡏⢀⡄⠸⣿⡏⠉⣿⠉⢉⣩⠉⢿⡟⠉⢻⡿⠉⡏⣉⣉⠙⢏⠉⢿⡿⠉⡝⠉⣉⡉⠻⡇⠀⣶⣶⠄⢸⡇⠸⠿⠗⠀⣼⣧⡀⠉⢀⣾⣿⣿⣿⣿
⣿⣿⣿⣿⡿⠁⢠⡿⡙⣿⣿⣆⠈⢿⡿⣿⣿⣿⣿⡟⠀⣈⣉⡀⠹⡇⠀⣿⠀⣿⣿⡆⠘⠀⣄⠈⠁⣸⠋⢡⣤⠀⢸⣆⠘⠁⣰⡄⠠⣤⡤⢤⡇⠀⣀⣀⣤⣾⡇⢰⣶⣶⠀⢸⡟⠁⣀⠈⢿⣿⣿⣿⣿
⣿⣿⣿⣿⠁⢠⣭⣣⣿⡸⣱⣶⣆⠀⠀⠘⣿⣿⣿⣤⣼⣿⣿⣷⣤⣥⣤⣿⣤⣿⣿⣿⣤⣼⣿⣦⣴⣿⣦⣤⣤⣤⣼⣿⣤⣴⣿⣷⣦⣤⣤⣾⣧⣤⣿⣿⣿⣿⣧⣤⣤⣤⣴⣮⣤⣾⣿⣧⣤⣽⣿⣿⣿
⣿⣿⣿⠃⢀⣾⣿⣿⣿⣿⣿⣿⣷⡉⠲⣾⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿
⣿⣿⣧⣤⣾⣿⣿⣿⣿⣿⣿⣿⣿⣷⣤⣽⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿
EOF
    echo -e "${NC}"
    echo ""
    echo -e "${BLUE}═════════════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BLUE}                             ${BOLD}AirwavePBX${NC}"
    echo -e "${BLUE}                    ${BOLD}Professional Broadcasting PBX${NC}"
    echo -e "${BLUE}                           Version ${BOLD}${AIRWAVE_VERSION}${NC}"
    echo -e "${BLUE}              ${BOLD}The Ultimate Phone System for Radio Stations${NC}"
    echo -e "${BLUE}═════════════════════════════════════════════════════════════════════════${NC}"
    echo ""
}

# Logging functions
print_step() {
    echo -e "\n${MAGENTA}▶${NC} ${BOLD}$1${NC}"
}

print_info() {
    echo -e "  ${BLUE}ℹ${NC} $1"
}

print_success() {
    echo -e "  ${GREEN}✓${NC} $1"
}

print_error() {
    echo -e "  ${RED}✗${NC} $1"
}

print_warning() {
    echo -e "  ${YELLOW}⚠${NC} $1"
}

# Pause for user to read important messages
pause_if_needed() {
    if [[ -t 0 ]]; then  # Check if running interactively
        echo ""
        read -p "Press Enter to continue..." -r
        echo ""
    fi
}

# Check if running as root
check_root() {
    if [[ $EUID -ne 0 ]]; then
        print_error "This script must be run as root"
        exit 1
    fi
    
    # Configure any pending packages non-interactively
    print_info "Configuring any pending packages..."
    
    # Show what we're doing
    echo -e "${CYAN}  → Running: dpkg --configure -a${NC}"
    dpkg --configure -a 2>&1 | grep -v "already installed and configured" || true
    
    # Clear any apt locks
    print_info "Clearing any package locks..."
    rm -f /var/lib/apt/lists/lock 2>/dev/null || true
    rm -f /var/cache/apt/archives/lock 2>/dev/null || true
    rm -f /var/lib/dpkg/lock* 2>/dev/null || true
}

# Detect OS
detect_os() {
    print_step "Detecting Operating System"
    
    if [[ -f /etc/os-release ]]; then
        . /etc/os-release
        OS=$ID
        VER=$VERSION_ID
        print_info "Detected: $PRETTY_NAME"
    else
        print_error "Cannot detect OS"
        exit 1
    fi
    
    if [[ "$OS" == "ubuntu" ]] && [[ "$VER" == "24.04" ]]; then
        print_success "Ubuntu 24.04 LTS detected - Perfect!"
    elif [[ "$OS" == "debian" ]] && [[ "$VER" == "12" ]]; then
        print_warning "Debian 12 detected - Will work but Ubuntu 24.04 LTS recommended"
    else
        print_error "This installer requires Ubuntu 24.04 LTS or Debian 12"
        read -p "Continue anyway? [y/N] " -n 1 -r
        echo
        if [[ ! $REPLY =~ ^[Yy]$ ]]; then
            exit 1
        fi
    fi
}

# Backup existing data
backup_existing_data() {
    if [[ -d "$INSTALL_DIR" ]] || [[ -d "$SSL_DIR" ]]; then
        print_step "Backing Up Existing Data"
        
        # Create backup directory
        mkdir -p "$BACKUP_DIR"
        local backup_name="backup-$(date +%Y%m%d-%H%M%S)"
        local backup_path="$BACKUP_DIR/$backup_name"
        mkdir -p "$backup_path"
        
        # Backup SSL certificates if they exist
        if [[ -d "$SSL_DIR/live" ]]; then
            print_info "Backing up SSL certificates..."
            cp -r "$SSL_DIR" "$backup_path/"
            print_success "SSL certificates backed up"
        fi
        
        # Backup data directory
        if [[ -d "$DATA_DIR" ]]; then
            print_info "Backing up data directory..."
            cp -r "$DATA_DIR" "$backup_path/"
            print_success "Data directory backed up"
        fi
        
        # Backup configuration
        if [[ -f "$INSTALL_DIR/config.yaml" ]]; then
            print_info "Backing up configuration..."
            cp "$INSTALL_DIR/config.yaml" "$backup_path/"
            print_success "Configuration backed up"
        fi
        
        print_success "Backup completed: $backup_path"
    fi
}

# Restore SSL certificates
restore_ssl_certificates() {
    # Find most recent backup with SSL certificates
    local latest_ssl_backup=""
    for backup in $(ls -t "$BACKUP_DIR" 2>/dev/null); do
        if [[ -d "$BACKUP_DIR/$backup/letsencrypt/live" ]]; then
            latest_ssl_backup="$BACKUP_DIR/$backup/letsencrypt"
            break
        fi
    done
    
    if [[ -n "$latest_ssl_backup" ]]; then
        print_info "Restoring SSL certificates from backup..."
        cp -r "$latest_ssl_backup"/* "$SSL_DIR/"
        print_success "SSL certificates restored"
        return 0
    fi
    
    return 1
}

# Install system dependencies
install_dependencies() {
    print_step "Installing System Dependencies"
    
    # Create log directory
    mkdir -p /var/log/airwavepbx
    local LOG_FILE="/var/log/airwavepbx/install-$(date +%Y%m%d-%H%M%S).log"
    
    # Function to run apt commands with proper error handling
    run_apt() {
        local cmd="$1"
        local desc="$2"
        
        print_info "$desc"
        echo "Running: $cmd" >> $LOG_FILE
        
        # Show the actual command being run
        echo -e "${CYAN}  → Running: $cmd${NC}"
        
        # Run the command and capture output
        eval "$cmd" 2>&1 | tee -a $LOG_FILE
        local exit_code=${PIPESTATUS[0]}
        
        if [[ $exit_code -ne 0 ]]; then
            print_warning "$desc failed (exit code: $exit_code)"
            ((INSTALL_ERRORS++))
            return $exit_code
        fi
        
        return 0
    }
    
    # Fix any broken dependencies first
    print_info "Checking and fixing package dependencies..."
    run_apt "dpkg --configure -a" "Configuring pending packages"
    run_apt "apt-get install -f -y" "Fixing broken dependencies"
    
    # Update package lists
    run_apt "apt-get update" "Updating package lists" || {
        print_error "Failed to update package lists. Check your internet connection."
        return 1
    }
    
    # Upgrade packages with visible output
    print_info "Upgrading system packages (this may take a few minutes)..."
    print_info "Note: We will preserve your SSH and other configurations"
    
    # Run upgrade with full output so user can see what's happening
    apt-get upgrade -y \
        -o Dpkg::Options::="--force-confold" \
        -o Dpkg::Options::="--force-confdef" \
        2>&1 | tee -a $LOG_FILE || {
        print_warning "Some packages failed to upgrade. This is usually OK."
        print_info "Check $LOG_FILE for details"
    }
    
    # Install essential packages
    local packages=(
        # Build tools
        build-essential
        git
        curl
        wget
        
        # FreeSWITCH dependencies
        autoconf
        automake
        libtool
        pkg-config
        libtool-bin
        
        # Audio libraries
        libspeex-dev
        libspeexdsp-dev
        libopus-dev
        libsndfile-dev
        
        # Database
        sqlite3
        libsqlite3-dev
        
        # SSL/Security
        openssl
        libssl-dev
        ca-certificates
        certbot
        python3-certbot
        
        # System utilities
        systemd
        rsyslog
        logrotate
        htop
        net-tools
        dnsutils
        
        # Firewall
        ufw
        fail2ban
    )
    
    # Install packages
    print_info "Installing required packages (${#packages[@]} total)..."
    local failed_packages=()
    
    for package in "${packages[@]}"; do
        echo -ne "${CYAN}  → Installing: ${package}...${NC}"
        
        if apt-get install -y \
            -o Dpkg::Options::="--force-confold" \
            -o Dpkg::Options::="--force-confdef" \
            $package >> $LOG_FILE 2>&1; then
            echo -e " ${GREEN}✓${NC}"
        else
            echo -e " ${RED}✗${NC}"
            failed_packages+=("$package")
            print_warning "Failed to install $package (will retry later)"
        fi
    done
    
    # Retry failed packages
    if [[ ${#failed_packages[@]} -gt 0 ]]; then
        print_info "Retrying failed packages..."
        apt-get install -f -y >> $LOG_FILE 2>&1
        
        for package in "${failed_packages[@]}"; do
            echo -ne "${CYAN}  → Retry: ${package}...${NC}"
            if apt-get install -y $package >> $LOG_FILE 2>&1; then
                echo -e " ${GREEN}✓${NC}"
            else
                echo -e " ${RED}✗${NC}"
                print_error "Could not install $package - this may cause issues"
                ((INSTALL_ERRORS++))
            fi
        done
    fi
    
    print_success "Dependencies installation completed"
    if [[ $INSTALL_ERRORS -gt 0 ]]; then
        print_warning "There were $INSTALL_ERRORS errors during dependency installation"
        print_info "The installer will continue, but some features might not work"
        print_info "Check $LOG_FILE for details"
    fi
}

# Install Asterisk with all modules and codecs
install_asterisk() {
    print_step "Installing Asterisk PBX System"
    
    local LOG_FILE="/var/log/airwavepbx/install-asterisk-$(date +%Y%m%d-%H%M%S).log"
    
    # Clean up any old repository configurations
    print_info "Cleaning up old repository configurations..."
    rm -f /etc/apt/sources.list.d/freeswitch*.list 2>/dev/null
    rm -f /etc/apt/sources.list.d/signalwire*.list 2>/dev/null
    
    # Update package lists
    print_info "Updating package lists..."
    apt-get update >> $LOG_FILE 2>&1 || true
    
    # Install Asterisk and ALL available modules
    print_info "Installing Asterisk 20 with all modules and codecs..."
    
    local asterisk_packages=(
        # Core Asterisk
        asterisk
        asterisk-core-sounds-en
        asterisk-core-sounds-en-wav
        asterisk-core-sounds-en-gsm
        asterisk-core-sounds-en-g722
        
        # Additional sounds and prompts
        asterisk-moh-opsound-wav
        asterisk-moh-opsound-g722
        asterisk-prompt-it
        
        # Modules and codecs
        asterisk-modules
        asterisk-mp3         # MP3 support
        asterisk-opus        # Opus codec (best quality)
        asterisk-espeak      # Text-to-speech
        asterisk-flite       # Alternative TTS
        asterisk-voicemail   # Voicemail support
        
        # Database connectors
        asterisk-mysql       # MySQL CDR support
        asterisk-postgresql  # PostgreSQL CDR support
        asterisk-odbc        # ODBC support
        
        # Additional protocols
        asterisk-ooh323      # H.323 support
        asterisk-mobile      # Bluetooth support
        asterisk-dahdi       # Hardware card support
        
        # Development files (for AGI scripts)
        asterisk-dev
        asterisk-doc
        
        # Additional utilities
        sox                  # Audio conversion
        mpg123              # MP3 playback
        ffmpeg              # Media conversion
        lame                # MP3 encoding
        
        # SIP testing tools
        sipsak              # SIP testing
        sngrep              # SIP packet capture
    )
    
    echo -e "${CYAN}  → Installing Asterisk packages...${NC}"
    for package in "${asterisk_packages[@]}"; do
        echo -ne "${CYAN}  → Installing ${package}...${NC} "
        if apt-get install -y -o Dpkg::Options::="--force-confold" $package >> $LOG_FILE 2>&1; then
            echo -e "${GREEN}✓${NC}"
        else
            # Check if it's a critical package
            case $package in
                asterisk|asterisk-modules)
                    echo -e "${RED}✗ FAILED${NC}"
                    print_error "Failed to install critical package: $package"
                    ((INSTALL_ERRORS++))
                    ;;
                *)
                    echo -e "${YELLOW}⚠ (not found)${NC}"
                    print_warning "Optional package not available: $package" >> $LOG_FILE
                    ;;
            esac
        fi
    done
    
    # Create Asterisk user if needed
    useradd -r -d /var/lib/asterisk -s /usr/sbin/nologin -c "Asterisk" asterisk 2>/dev/null || true
    
    # Set up directories with proper permissions
    for dir in /var/lib/asterisk /var/log/asterisk /var/run/asterisk /var/spool/asterisk /etc/asterisk; do
        mkdir -p $dir
        chown -R asterisk:asterisk $dir
    done
    
    # Enable Asterisk service
    systemctl enable asterisk >> $LOG_FILE 2>&1
    systemctl stop asterisk &>/dev/null || true
    
    # Check if core packages were installed
    if command -v asterisk &> /dev/null; then
        print_success "Asterisk installed successfully"
        print_info "Note: Some optional packages may not be available in Ubuntu 24.04"
        print_info "Core functionality including Opus codec is built into main Asterisk package"
    else
        print_error "Asterisk installation failed!"
        return 1
    fi
    
    return 0
}

# Install Asterisk as fallback when FreeSWITCH is not available
install_asterisk_fallback() {
    print_warning "Installing Asterisk instead of FreeSWITCH..."
    
    # Install Asterisk from Ubuntu repositories
    print_info "Installing Asterisk packages..."
    local asterisk_packages=(
        asterisk
        asterisk-core-sounds-en
        asterisk-core-sounds-en-wav
        asterisk-modules
        asterisk-opus
        asterisk-mp3
    )
    
    for package in "${asterisk_packages[@]}"; do
        echo -ne "${CYAN}  → Installing ${package}...${NC} "
        if apt-get install -y -o Dpkg::Options::="--force-confold" $package >> $LOG_FILE 2>&1; then
            echo -e "${GREEN}✓${NC}"
        else
            echo -e "${YELLOW}⚠${NC}"
        fi
    done
    
    # Create Asterisk user if needed
    useradd -r -d /var/lib/asterisk -s /usr/sbin/nologin -c "Asterisk" asterisk 2>/dev/null || true
    
    # Set up directories
    for dir in /var/lib/asterisk /var/log/asterisk /var/run/asterisk /etc/asterisk; do
        mkdir -p $dir
        chown -R asterisk:asterisk $dir
    done
    
    # Enable Asterisk service
    systemctl enable asterisk >> $LOG_FILE 2>&1
    systemctl stop asterisk &>/dev/null || true
    
    # Set a flag that we're using Asterisk
    echo "asterisk" > /tmp/pbx_type
    
    print_success "Asterisk installed from Ubuntu repositories"
    return 0
}


# Configure Asterisk for broadcasting
configure_asterisk() {
    print_step "Configuring Asterisk for Broadcasting"
    
    # Backup original configs
    print_info "Backing up original configuration..."
    [ -d /etc/asterisk ] && cp -r /etc/asterisk /etc/asterisk.backup || true
    
    # Create main configuration files
    print_info "Creating Asterisk configuration..."
    
    # Configure AMI for API access
    print_info "Configuring Asterisk Manager Interface (AMI)..."
    cat > /etc/asterisk/manager.conf << EOF
[general]
enabled = yes
port = 5038
bindaddr = 127.0.0.1
displayconnects = yes

[airwave]
secret = $ESL_PASSWORD
deny = 0.0.0.0/0.0.0.0
permit = 127.0.0.1/255.255.255.255
read = all
write = all
writetimeout = 5000
EOF
    
    # Configure PJSIP for modern SIP
    print_info "Configuring PJSIP..."
    cat > /etc/asterisk/pjsip.conf << EOF
[global]
type=global
; Custom User-Agent sent to all SIP providers
user_agent=AirwavePBX (Asterisk)
default_realm=airwavepbx.local

[transport-udp]
type=transport
protocol=udp
bind=0.0.0.0:5060
local_net=10.0.0.0/8
local_net=172.16.0.0/12
local_net=192.168.0.0/16

[transport-tcp]
type=transport
protocol=tcp
bind=0.0.0.0:5060

; Template for VoIP providers
[provider-template](!)
type=registration
transport=transport-udp
retry_interval=60
max_retries=10
auth_rejection_permanent=no
contact_user=s

[provider-endpoint-template](!)
type=endpoint
transport=transport-udp
context=from-trunk
disallow=all
allow=ulaw
allow=alaw
allow=g722
allow=opus
allow=gsm
dtmf_mode=rfc4733
direct_media=no
force_rport=yes
rewrite_contact=yes
rtp_symmetric=yes
timers=yes

; Callcentric template
[callcentric-auth](provider-template)
type=auth
auth_type=userpass
username=YOUR_CALLCENTRIC_USERNAME
password=YOUR_CALLCENTRIC_PASSWORD

[callcentric-reg](provider-template)
type=registration
server_uri=sip:callcentric.com
client_uri=sip:YOUR_CALLCENTRIC_USERNAME@callcentric.com
outbound_auth=callcentric-auth

[callcentric](provider-endpoint-template)
type=endpoint
outbound_auth=callcentric-auth
aors=callcentric

[callcentric]
type=aor
contact=sip:callcentric.com

; VoIP.ms template
[voipms-auth](provider-template)
type=auth
auth_type=userpass
username=YOUR_VOIPMS_USERNAME
password=YOUR_VOIPMS_PASSWORD

[voipms-reg](provider-template)
type=registration
server_uri=sip:YOUR_SERVER.voip.ms
client_uri=sip:YOUR_VOIPMS_USERNAME@YOUR_SERVER.voip.ms
outbound_auth=voipms-auth

[voipms](provider-endpoint-template)
type=endpoint
outbound_auth=voipms-auth
aors=voipms

[voipms]
type=aor
contact=sip:YOUR_SERVER.voip.ms
EOF
    
    # Configure extensions
    print_info "Configuring extensions..."
    cat > /etc/asterisk/extensions.conf << EOF
[general]
static=yes
writeprotect=no
clearglobalvars=no

[globals]
TRUNK_OPTIONS=g

[from-trunk]
; Incoming calls from trunks
exten => _X.,1,NoOp(Incoming call from \${CALLERID(num)} to \${EXTEN})
 same => n,Goto(incoming,\${EXTEN},1)

[incoming]
; Main incoming call handler
exten => _X.,1,Answer()
 same => n,Wait(1)
 same => n,Playback(welcome)
 same => n,Goto(main-menu,s,1)

[main-menu]
; IVR for incoming calls
exten => s,1,Background(press-1-for-sales)
 same => n,WaitExten(10)

exten => 1,1,Goto(queue-sales,s,1)
exten => 2,1,Goto(queue-support,s,1)

[internal]
; Internal extensions
exten => _1XXX,1,Dial(PJSIP/\${EXTEN},30)
 same => n,Voicemail(\${EXTEN})
 same => n,Hangup()

[outbound]
; Outbound calling
exten => _NXXNXXXXXX,1,NoOp(Outbound call to \${EXTEN})
 same => n,Set(CALLERID(all)=YOUR_OUTBOUND_CID)
 same => n,Dial(PJSIP/\${EXTEN}@voipms,60)
 same => n,Hangup()

[conference]
; Conference rooms for broadcasting
exten => 3000,1,Answer()
 same => n,ConfBridge(1,default_bridge,default_user)

exten => 3001,1,Answer()
 same => n,ConfBridge(2,default_bridge,default_user)
EOF
    
    # Configure modules to load
    print_info "Configuring modules..."
    cat > /etc/asterisk/modules.conf << EOF
[modules]
autoload=yes
; Explicitly load important modules
load => app_bridgewait.so
load => app_confbridge.so
load => app_mixmonitor.so
load => app_originate.so
load => app_park.so
load => app_playback.so
load => app_queue.so
load => app_voicemail.so
load => chan_pjsip.so
load => codec_opus.so       ; May be built-in or separate module
load => format_mp3.so
load => res_agi.so
load => res_config_mysql.so
load => res_config_pgsql.so
load => res_musiconhold.so
EOF
    
    # Configure RTP settings
    print_info "Configuring RTP..."
    cat > /etc/asterisk/rtp.conf << EOF
[general]
rtpstart=16384
rtpend=32768
rtcpinterval=5000
strictrtp=yes
EOF
    
    # Configure music on hold for radio
    print_info "Configuring music on hold..."
    cat > /etc/asterisk/musiconhold.conf << EOF
[default]
mode=files
directory=/var/lib/asterisk/moh

[radio-hold]
mode=custom
application=/usr/bin/mpg123 -q -r 8000 -f 8192 --mono -s
EOF
    
    # Set permissions
    chown -R asterisk:asterisk /etc/asterisk
    chmod 640 /etc/asterisk/*.conf
    
    print_success "Asterisk configured for broadcasting"
}

# Install AirwavePBX
install_airwavepbx() {
    print_step "Installing AirwavePBX"
    
    local LOG_FILE="/var/log/airwavepbx/install-airwavepbx-$(date +%Y%m%d-%H%M%S).log"
    
    # Create directories
    print_info "Creating directories..."
    mkdir -p $INSTALL_DIR $DATA_DIR $LOG_DIR
    
    # Install Go if not present
    if ! command -v go &> /dev/null; then
        print_info "Installing Go compiler..."
        apt-get install -y golang-go >> $LOG_FILE 2>&1 || {
            print_error "Failed to install Go"
            ((INSTALL_ERRORS++))
            return 1
        }
    fi
    
    # Build AirwavePBX binary
    print_info "Building AirwavePBX binary..."
    
    # Check for source code in different locations
    if [ -d "src" ] && [ -f "src/go.mod" ]; then
        cd src
    elif [ -f "go.mod" ]; then
        # Source is in current directory
        echo "Building from current directory..." >> $LOG_FILE
    else
        print_error "Source code not found (no go.mod file)"
        ((INSTALL_ERRORS++))
        return 1
    fi
    
    # Build the binary
    print_info "Running go build (this may take a minute)..."
    go mod download >> $LOG_FILE 2>&1
    GOOS=linux GOARCH=amd64 go build -ldflags="-s -w -X main.Version=$AIRWAVE_VERSION" -o airwavepbx ./cmd >> $LOG_FILE 2>&1
    
    if [ $? -eq 0 ] && [ -f "airwavepbx" ]; then
        cp airwavepbx $INSTALL_DIR/
        chmod +x $INSTALL_DIR/airwavepbx
        print_success "AirwavePBX binary built successfully"
    else
        print_error "Failed to build AirwavePBX - check $LOG_FILE for details"
        ((INSTALL_ERRORS++))
        return 1
    fi
    
    # Return to original directory if we changed
    [ -d "../" ] && cd .. || true
    
    # Copy web assets
    print_info "Installing web interface..."
    if [ -d "src/web" ]; then
        cp -r src/web $INSTALL_DIR/
    elif [ -d "web" ]; then
        cp -r web $INSTALL_DIR/
    fi
    
    # Copy FAP icons (renamed from fontawesome-pro)
    print_info "Installing icon assets..."
    if [ -d "fap" ]; then
        mkdir -p $INSTALL_DIR/web/static/
        cp -r fap $INSTALL_DIR/web/static/
    fi
    
    # Copy logo assets
    if [ -d "logos" ]; then
        cp -r logos $INSTALL_DIR/web/static/
    fi
    
    # Create configuration file
    print_info "Creating configuration..."
    cat > $INSTALL_DIR/config.yaml << EOF
# AirwavePBX Configuration
version: $AIRWAVE_VERSION

server:
  host: 0.0.0.0
  port: 8080
  tls_enabled: true
  tls_cert: /etc/letsencrypt/live/\${DOMAIN}/fullchain.pem
  tls_key: /etc/letsencrypt/live/\${DOMAIN}/privkey.pem

database:
  path: $DATA_DIR/airwavepbx.db
  
asterisk:
  ami_host: 127.0.0.1
  ami_port: 5038
  ami_username: airwave
  ami_password: $ESL_PASSWORD
  
api:
  key: $API_KEY
  cors_origins:
    - https://itsyourradio.com
    - https://broadcaster.itsyourradio.com
    
logging:
  level: info
  path: $LOG_DIR/airwavepbx.log
  max_size: 100
  max_backups: 7
  
admin:
  username: admin
  password: $ADMIN_PASSWORD
EOF
    
    # Set permissions
    chmod 600 $INSTALL_DIR/config.yaml
    
    print_success "AirwavePBX installed"
}

# Create systemd service
create_systemd_service() {
    print_step "Creating System Service"
    
    cat > /etc/systemd/system/$SYSTEMD_SERVICE << EOF
[Unit]
Description=AirwavePBX Professional Broadcasting PBX
After=network.target asterisk.service
Requires=asterisk.service

[Service]
Type=simple
User=root
Group=root
WorkingDirectory=$INSTALL_DIR
ExecStart=$INSTALL_DIR/airwavepbx
Restart=always
RestartSec=10
StandardOutput=append:$LOG_DIR/airwavepbx.log
StandardError=append:$LOG_DIR/airwavepbx.log

# Security
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF
    
    systemctl daemon-reload
    systemctl enable $SYSTEMD_SERVICE
    
    print_success "System service created"
}

# Configure firewall
configure_firewall() {
    print_step "Configuring Firewall (UFW)"
    
    # Check if UFW is installed
    if ! command -v ufw &> /dev/null; then
        print_error "UFW not installed - skipping firewall configuration"
        return 1
    fi
    
    print_info "Opening required ports for PBX operation..."
    
    # Enable UFW
    echo -e "${CYAN}  → Enabling UFW firewall...${NC}"
    ufw --force enable
    
    # Allow SSH (preserve existing SSH rules)
    echo -e "${CYAN}  → Allowing SSH (port 22)...${NC}"
    ufw allow ssh
    
    # Web ports
    echo -e "${CYAN}  → Opening web ports...${NC}"
    ufw allow 80/tcp   # HTTP for Let's Encrypt
    ufw allow 443/tcp  # HTTPS for web interface
    ufw allow 8080/tcp # AirwavePBX web interface
    
    # SIP signaling ports
    echo -e "${CYAN}  → Opening SIP signaling ports...${NC}"
    ufw allow 5060/udp # SIP UDP
    ufw allow 5060/tcp # SIP TCP
    ufw allow 5061/tcp # SIP TLS
    
    # RTP media ports
    echo -e "${CYAN}  → Opening RTP media ports (16384-32768/udp)...${NC}"
    ufw allow 16384:32768/udp # RTP Media Stream
    
    # AMI port (local only - no rule needed)
    echo -e "${CYAN}  → Keeping AMI port 5038 local only${NC}"
    
    # WebRTC ports if needed
    echo -e "${CYAN}  → Opening WebRTC ports...${NC}"
    ufw allow 3478/tcp # STUN/TURN TCP
    ufw allow 3478/udp # STUN/TURN UDP
    
    # Reload firewall
    echo -e "${CYAN}  → Reloading firewall rules...${NC}"
    ufw reload
    
    # Show status
    echo ""
    print_info "Current firewall status:"
    ufw status numbered | head -20
    
    print_success "Firewall configured successfully"
}

# Configure fail2ban for security
configure_fail2ban() {
    print_step "Configuring Fail2ban Security"
    
    # Check if fail2ban is installed
    if ! command -v fail2ban-client &> /dev/null; then
        print_error "Fail2ban not installed - skipping security configuration"
        return 1
    fi
    
    print_info "Setting up intrusion prevention..."
    
    # Create Asterisk fail2ban filter
    echo -e "${CYAN}  → Creating Asterisk filter...${NC}"
    cat > /etc/fail2ban/filter.d/asterisk.conf << 'EOF'
[Definition]
failregex = NOTICE.* .*: Registration from '.*' failed for '<HOST>:.*' - Wrong password
            NOTICE.* .*: Registration from '.*' failed for '<HOST>:.*' - No matching peer found
            NOTICE.* .*: Registration from '.*' failed for '<HOST>:.*' - Username/auth name mismatch
            NOTICE.* .*: Registration from '.*' failed for '<HOST>:.*' - Device does not match ACL
            NOTICE.* <HOST> failed to authenticate as '.*'$
            NOTICE.* .*: No registration for peer '.*' \(from <HOST>\)
            NOTICE.* .*: Host <HOST> failed MD5 authentication
            NOTICE.* .*: Failed to authenticate user .*@<HOST>.*
            SECURITY.* SecurityEvent="InvalidPassword".*RemoteAddress="IPV[46]/(UDP|TCP|TLS)/<HOST>/.*".*
            
ignoreregex =
EOF

    # Create Asterisk jail configuration
    echo -e "${CYAN}  → Configuring Asterisk jail...${NC}"
    cat > /etc/fail2ban/jail.d/asterisk.conf << 'EOF'
[asterisk]
enabled = true
port = 5060,5061
protocol = udp,tcp
filter = asterisk
logpath = /var/log/asterisk/security.log
maxretry = 5
findtime = 600
bantime = 3600
action = %(banaction)s[name=%(__name__)s-tcp, port="%(port)s", protocol="tcp", chain="%(chain)s", actname=%(banaction)s-tcp]
         %(banaction)s[name=%(__name__)s-udp, port="%(port)s", protocol="udp", chain="%(chain)s", actname=%(banaction)s-udp]
EOF

    # Create AirwavePBX API protection
    echo -e "${CYAN}  → Creating AirwavePBX API protection...${NC}"
    cat > /etc/fail2ban/filter.d/airwavepbx.conf << 'EOF'
[Definition]
failregex = ^.*Failed login attempt.*from <HOST>.*$
            ^.*Unauthorized API access.*from <HOST>.*$
            ^.*Invalid API key.*from <HOST>.*$

ignoreregex =

datepattern = %%Y-%%m-%%d %%H:%%M:%%S
EOF

    # Create AirwavePBX jail
    cat > /etc/fail2ban/jail.d/airwavepbx.conf << 'EOF'
[airwavepbx]
enabled = true
port = 8080,80,443
protocol = tcp
filter = airwavepbx
logpath = /var/log/airwavepbx/airwavepbx.log
maxretry = 5
findtime = 600
bantime = 3600
EOF

    # Configure SSH protection (usually already there but ensure it's enabled)
    echo -e "${CYAN}  → Ensuring SSH protection is enabled...${NC}"
    cat > /etc/fail2ban/jail.d/sshd.conf << 'EOF'
[sshd]
enabled = true
port = ssh
filter = sshd
logpath = /var/log/auth.log
maxretry = 3
findtime = 600
bantime = 3600
EOF

    # Enable and start fail2ban
    echo -e "${CYAN}  → Starting fail2ban service...${NC}"
    systemctl enable fail2ban
    systemctl start fail2ban
    
    # Wait for fail2ban to start
    sleep 2
    
    # Check if fail2ban is running
    if systemctl is-active --quiet fail2ban; then
        # Show status
        echo ""
        print_info "Current fail2ban status:"
        fail2ban-client status 2>/dev/null | head -10 || print_warning "Fail2ban is starting up..."
        print_success "Fail2ban security configured successfully"
    else
        print_warning "Fail2ban service failed to start - check logs with: journalctl -u fail2ban"
    fi
}

# Configure log rotation
configure_logging() {
    print_step "Configuring Log Rotation"
    
    cat > /etc/logrotate.d/airwavepbx << EOF
$LOG_DIR/*.log {
    daily
    missingok
    rotate 14
    compress
    notifempty
    create 0640 root root
    sharedscripts
    postrotate
        systemctl reload airwavepbx > /dev/null 2>&1 || true
    endscript
}

/var/log/asterisk/*.log {
    daily
    missingok
    rotate 7
    compress
    notifempty
    create 0640 asterisk asterisk
    sharedscripts
    postrotate
        /usr/sbin/asterisk -rx 'logger reload' > /dev/null 2>&1 || true
    endscript
}
EOF
    
    print_success "Log rotation configured"
}

# Generate self-signed certificate
generate_self_signed_cert() {
    local domain=$1
    
    print_info "Generating self-signed certificate for $domain..."
    
    # Create SSL directory
    mkdir -p $INSTALL_DIR/ssl
    
    # Generate private key and certificate
    openssl req -x509 -nodes -days 365 -newkey rsa:2048 \
        -keyout $INSTALL_DIR/ssl/privkey.pem \
        -out $INSTALL_DIR/ssl/fullchain.pem \
        -subj "/C=US/ST=State/L=City/O=AirwavePBX/CN=$domain" \
        2>/dev/null
    
    chmod 600 $INSTALL_DIR/ssl/*
    
    # Update configuration
    sed -i "s/\${DOMAIN}/$domain/g" $INSTALL_DIR/config.yaml
    
    print_success "Self-signed certificate generated"
    print_warning "Browsers will show a security warning for self-signed certificates"
}

# Check for existing valid Let's Encrypt certificates
check_existing_letsencrypt_certs() {
    local cert_found=false
    
    if [[ -d "$SSL_DIR/live" ]]; then
        for cert_dir in "$SSL_DIR/live"/*; do
            if [[ -d "$cert_dir" && "$(basename "$cert_dir")" != "README" ]]; then
                local domain=$(basename "$cert_dir")
                local cert_file="$cert_dir/fullchain.pem"
                local key_file="$cert_dir/privkey.pem"
                
                # Check if certificate files exist and are valid
                if [[ -f "$cert_file" && -f "$key_file" ]]; then
                    # Check if certificate is not expired and not self-signed
                    local expiry_date=$(openssl x509 -enddate -noout -in "$cert_file" 2>/dev/null | cut -d= -f2)
                    local is_self_signed=$(openssl x509 -noout -issuer -subject -in "$cert_file" 2>/dev/null | grep -c "CN=localhost")
                    
                    if [[ -n "$expiry_date" && $is_self_signed -eq 0 ]]; then
                        local expiry_epoch=$(date -d "$expiry_date" +%s 2>/dev/null)
                        local current_epoch=$(date +%s)
                        local days_until_expiry=$(( ($expiry_epoch - $current_epoch) / 86400 ))
                        
                        # If certificate is valid for more than 7 days, use it
                        if [[ $days_until_expiry -gt 7 ]]; then
                            print_success "Found valid Let's Encrypt certificate for $domain (expires in $days_until_expiry days)"
                            DOMAIN="$domain"
                            
                            # Copy certificates to AirwavePBX readable location
                            mkdir -p $INSTALL_DIR/ssl
                            cp "$cert_file" $INSTALL_DIR/ssl/fullchain.pem
                            cp "$key_file" $INSTALL_DIR/ssl/privkey.pem
                            chmod 600 $INSTALL_DIR/ssl/*
                            
                            # Update configuration with domain
                            sed -i "s/\${DOMAIN}/$DOMAIN/g" $INSTALL_DIR/config.yaml
                            
                            cert_found=true
                            break
                        else
                            print_warning "Found expired Let's Encrypt certificate for $domain (expires in $days_until_expiry days)"
                        fi
                    fi
                fi
            fi
        done
    fi
    
    if $cert_found; then
        return 0
    else
        return 1
    fi
}

# Configure SSL with Let's Encrypt
configure_ssl() {
    print_step "Configuring SSL Certificate"
    
    # First check for existing valid Let's Encrypt certificates
    if check_existing_letsencrypt_certs; then
        print_info "Using existing valid Let's Encrypt certificate for $DOMAIN"
        setup_ssl_renewal
        return
    fi
    
    # Check if we restored certificates from backup
    if restore_ssl_certificates; then
        print_success "Using restored SSL certificates from backup"
        setup_ssl_renewal
        return
    fi
    
    # Ask for domain configuration
    print_info "SSL Certificate Setup"
    echo ""
    read -p "Enter your domain name (e.g., pbx.yourdomain.com): " DOMAIN
    
    if [[ -z "$DOMAIN" ]]; then
        print_warning "No domain provided - generating self-signed certificate for localhost"
        DOMAIN="localhost"
        generate_self_signed_cert "$DOMAIN"
        return
    fi
    
    # Ask if they want to use standard web ports
    echo ""
    print_info "Port Configuration"
    echo "By default, AirwavePBX runs on port 8080."
    echo "Would you like to run on standard web ports instead?"
    echo "This will make your site accessible at https://$DOMAIN (no port needed)"
    read -p "Use standard ports 80/443? [Y/n] " -n 1 -r
    echo
    
    if [[ $REPLY =~ ^[Yy]$ ]] || [[ -z $REPLY ]]; then
        USE_STANDARD_PORTS=true
        print_info "Will configure for standard ports 80/443"
        
        # Update configuration to use standard ports
        sed -i "s/port: 8080/port: 443/g" $INSTALL_DIR/config.yaml
        sed -i "s|host: 0.0.0.0|host: 0.0.0.0\n  redirect_http: true\n  http_port: 80|g" $INSTALL_DIR/config.yaml
    else
        USE_STANDARD_PORTS=false
    fi
    
    # For Let's Encrypt, we need an email
    read -p "Enter your email address for Let's Encrypt notifications (or press Enter to skip): " EMAIL
    
    if [[ -z "$EMAIL" ]]; then
        print_warning "No email provided - using self-signed certificate"
        generate_self_signed_cert "$DOMAIN"
        return
    fi
    
    # Stop AirwavePBX temporarily to free port 80
    systemctl stop airwavepbx 2>/dev/null || true
    
    # Get certificate
    print_info "Obtaining SSL certificate from Let's Encrypt..."
    DEBIAN_FRONTEND=noninteractive certbot certonly \
        --standalone \
        --non-interactive \
        --agree-tos \
        --email "$EMAIL" \
        --domains "$DOMAIN" \
        --keep-until-expiring \
        --expand \
        --quiet
    
    if [[ $? -eq 0 ]]; then
        print_success "SSL certificate obtained successfully"
        
        # Update configuration with domain
        sed -i "s/\${DOMAIN}/$DOMAIN/g" $INSTALL_DIR/config.yaml
        
        # Copy certificates to AirwavePBX readable location
        mkdir -p $INSTALL_DIR/ssl
        cp "$SSL_DIR/live/$DOMAIN/fullchain.pem" $INSTALL_DIR/ssl/
        cp "$SSL_DIR/live/$DOMAIN/privkey.pem" $INSTALL_DIR/ssl/
        chmod 600 $INSTALL_DIR/ssl/*
        
        setup_ssl_renewal
    else
        print_error "Failed to obtain Let's Encrypt certificate"
        print_warning "Falling back to self-signed certificate..."
        
        # Generate self-signed certificate
        generate_self_signed_cert "$DOMAIN"
    fi
}

# Setup SSL auto-renewal
setup_ssl_renewal() {
    print_info "Setting up SSL auto-renewal..."
    
    # Create renewal hook script
    cat > /etc/letsencrypt/renewal-hooks/deploy/airwavepbx.sh << 'EOF'
#!/bin/bash
# Copy renewed certificates to AirwavePBX
DOMAIN=$(basename $(dirname $RENEWED_LINEAGE))
cp "$RENEWED_LINEAGE/fullchain.pem" /opt/airwavepbx/ssl/
cp "$RENEWED_LINEAGE/privkey.pem" /opt/airwavepbx/ssl/
chmod 600 /opt/airwavepbx/ssl/*
systemctl reload airwavepbx
EOF
    
    chmod +x /etc/letsencrypt/renewal-hooks/deploy/airwavepbx.sh
    
    # Test renewal
    certbot renew --dry-run &>/dev/null
    
    if [[ $? -eq 0 ]]; then
        print_success "SSL auto-renewal configured"
    else
        print_warning "SSL auto-renewal test failed - check configuration"
    fi
}

# Save credentials
save_credentials() {
    print_step "Saving Credentials"
    
    # Determine the access URL
    local WEB_URL
    local API_URL
    
    if [[ "$USE_STANDARD_PORTS" == "true" ]]; then
        WEB_URL="https://${DOMAIN:-YOUR_DOMAIN}"
        API_URL="https://${DOMAIN:-YOUR_DOMAIN}/api/v1"
    else
        WEB_URL="https://${DOMAIN:-YOUR_DOMAIN}:8080"
        API_URL="https://${DOMAIN:-YOUR_DOMAIN}:8080/api/v1"
    fi
    
    cat > $CREDENTIALS_FILE << EOF
════════════════════════════════════════════════════════════════════════
                        AirwavePBX Credentials
                    Generated: $(date '+%Y-%m-%d %H:%M:%S')
════════════════════════════════════════════════════════════════════════

Web Interface:
  URL:      $WEB_URL
  Username: admin
  Password: $ADMIN_PASSWORD

API Access:
  Endpoint: $API_URL
  API Key:  $API_KEY

FreeSWITCH ESL:
  Host:     127.0.0.1
  Port:     8021
  Password: $ESL_PASSWORD

Database:
  Type:     SQLite
  Path:     $DATA_DIR/airwavepbx.db

════════════════════════════════════════════════════════════════════════

IMPORTANT: Keep this file secure! It contains all your system passwords.
Location: $CREDENTIALS_FILE

EOF
    
    chmod 600 $CREDENTIALS_FILE
    print_success "Credentials saved to $CREDENTIALS_FILE"
}

# Start services
start_services() {
    print_step "Starting Services"
    
    print_info "Starting Asterisk..."
    systemctl start asterisk
    sleep 5
    
    print_info "Starting AirwavePBX..."
    systemctl start airwavepbx
    sleep 3
    
    # Check if services are running
    if systemctl is-active --quiet asterisk && systemctl is-active --quiet airwavepbx; then
        print_success "All services started successfully"
    else
        print_error "Services failed to start. Check logs:"
        print_info "  journalctl -u asterisk -n 50"
        print_info "  journalctl -u airwavepbx -n 50"
    fi
}

# Display completion message
display_completion() {
    echo ""
    echo -e "${GREEN}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
    echo -e "${GREEN}${BOLD}                    🎉 INSTALLATION COMPLETE! 🎉                        ${NC}"
    echo -e "${GREEN}${BOLD}                          AirwavePBX                                    ${NC}"
    echo -e "${GREEN}${BOLD}              Your Broadcasting PBX is Ready for Production              ${NC}"
    echo -e "${GREEN}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}Access Information:${NC}"
    
    # Show correct URL based on configuration
    if [[ "$USE_STANDARD_PORTS" == "true" ]]; then
        echo -e "  ${BOLD}Web Interface:${NC} https://${DOMAIN:-YOUR_DOMAIN}"
        echo -e "  ${BOLD}API Endpoint:${NC}  https://${DOMAIN:-YOUR_DOMAIN}/api/v1"
    else
        echo -e "  ${BOLD}Web Interface:${NC} https://${DOMAIN:-YOUR_DOMAIN}:8080"
        echo -e "  ${BOLD}API Endpoint:${NC}  https://${DOMAIN:-YOUR_DOMAIN}:8080/api/v1"
    fi
    
    echo -e "  ${BOLD}Username:${NC}      admin"
    echo -e "  ${BOLD}Password:${NC}      ${YELLOW}$ADMIN_PASSWORD${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}API Key:${NC}       ${YELLOW}$API_KEY${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}Important Files:${NC}"
    echo -e "  ${BOLD}Credentials:${NC}   $CREDENTIALS_FILE"
    echo -e "  ${BOLD}Configuration:${NC} $INSTALL_DIR/config.yaml"
    echo -e "  ${BOLD}Logs:${NC}          $LOG_DIR/"
    echo ""
    if [[ -d "$SSL_DIR/live" ]]; then
        echo -e "${GREEN}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
        echo -e "${GREEN}${BOLD}  ✓ SSL Certificate configured and active with auto-renewal enabled!   ${NC}"
        echo -e "${GREEN}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
    else
        echo -e "${RED}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
        echo -e "${RED}${BOLD}  IMPORTANT: Configure your domain and SSL certificate immediately!     ${NC}"
        echo -e "${RED}${BOLD}  Run: airwavepbx configure-ssl YOUR_DOMAIN.COM                       ${NC}"
        echo -e "${RED}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
    fi
    echo ""
}

# Installation summary
show_installation_summary() {
    echo ""
    print_step "Installation Summary"
    
    if [[ $INSTALL_ERRORS -eq 0 ]]; then
        print_success "Installation completed successfully with no errors!"
    else
        print_warning "Installation completed with $INSTALL_ERRORS errors"
        print_info "Some features may not work properly"
        print_info "Check logs in /var/log/airwavepbx/ for details"
    fi
    
    # Show service status
    echo ""
    print_info "Service Status:"
    for service in asterisk airwavepbx; do
        if systemctl is-active --quiet $service; then
            echo -e "  ${GREEN}✓${NC} $service is running"
        else
            echo -e "  ${RED}✗${NC} $service is not running"
        fi
    done
}

# Main installation flow
main() {
    print_banner
    check_root
    detect_os
    
    print_info "Starting AirwavePBX installation..."
    print_info "This will install Asterisk PBX, a Go API server, and configure your system"
    echo ""
    read -p "Continue with installation? [Y/n] " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Nn]$ ]]; then
        print_info "Installation cancelled"
        exit 0
    fi
    
    # Create installation log
    mkdir -p /var/log/airwavepbx
    MAIN_LOG="/var/log/airwavepbx/install-main-$(date +%Y%m%d-%H%M%S).log"
    
    print_info "Installation logs will be saved to: /var/log/airwavepbx/"
    echo ""
    
    # Backup existing data first
    backup_existing_data || true
    
    # Run each installation step and continue even if errors occur
    install_dependencies || print_warning "Dependencies installation had issues"
    echo ""
    
    install_asterisk || print_warning "Asterisk installation had issues"
    echo ""
    
    configure_asterisk || print_warning "Asterisk configuration had issues"
    echo ""
    
    install_airwavepbx || print_warning "AirwavePBX installation had issues"
    echo ""
    
    create_systemd_service || print_warning "Service creation had issues"
    echo ""
    
    configure_firewall || print_warning "Firewall configuration had issues"
    echo ""
    
    configure_fail2ban || print_warning "Fail2ban configuration had issues"
    echo ""
    
    configure_logging || print_warning "Logging configuration had issues"
    echo ""
    
    configure_ssl || print_warning "SSL configuration had issues"
    echo ""
    
    save_credentials || print_warning "Credential saving had issues"
    echo ""
    
    start_services || print_warning "Service startup had issues"
    echo ""
    
    # Show summary before completion
    show_installation_summary
    
    # Display completion screen
    display_completion
}

# Run main function
main "$@"