package api

// getAPIEndpoints returns API endpoint documentation
func getAPIEndpoints() map[string]interface{} {
	return map[string]interface{}{
		"auth": map[string]interface{}{
			"POST /auth/login": map[string]interface{}{
				"description": "Authenticate and receive JWT token",
				"body": map[string]interface{}{
					"username": "string",
					"password": "string",
				},
				"response": map[string]interface{}{
					"token": "string",
					"user":  "object",
				},
			},
			"POST /auth/logout": map[string]interface{}{
				"description": "Invalidate current session",
				"auth":        "required",
			},
			"GET /auth/me": map[string]interface{}{
				"description": "Get current user info",
				"auth":        "required",
			},
		},
		"calls": map[string]interface{}{
			"GET /calls/active": map[string]interface{}{
				"description": "List all active calls",
				"auth":        "required",
				"response":    "array of ActiveCall objects",
			},
			"GET /calls/history": map[string]interface{}{
				"description": "Get call history with pagination",
				"auth":        "required",
				"query": map[string]interface{}{
					"limit":  "number (default: 50)",
					"offset": "number (default: 0)",
				},
			},
			"POST /calls/:uuid/answer": map[string]interface{}{
				"description": "Answer an incoming call",
				"auth":        "required",
			},
			"POST /calls/:uuid/hangup": map[string]interface{}{
				"description": "End a call",
				"auth":        "required",
			},
			"POST /calls/:uuid/hold": map[string]interface{}{
				"description": "Put call on hold",
				"auth":        "required",
			},
			"POST /calls/:uuid/transfer": map[string]interface{}{
				"description": "Transfer call to another extension",
				"auth":        "required",
				"body": map[string]interface{}{
					"destination": "string (extension number)",
				},
			},
			"POST /calls/:uuid/conference": map[string]interface{}{
				"description": "Add call to conference",
				"auth":        "required",
				"body": map[string]interface{}{
					"conference_id": "string (default: 'on-air')",
				},
			},
			"POST /calls/:uuid/mute": map[string]interface{}{
				"description": "Mute a call",
				"auth":        "required",
			},
			"POST /calls/:uuid/on-air": map[string]interface{}{
				"description": "Set call on-air status",
				"auth":        "required",
				"body": map[string]interface{}{
					"on_air": "boolean",
				},
			},
			"POST /calls/dial": map[string]interface{}{
				"description": "Originate a new outbound call",
				"auth":        "required",
				"body": map[string]interface{}{
					"from": "string (caller ID)",
					"to":   "string (destination number)",
				},
			},
		},
		"extensions": map[string]interface{}{
			"GET /extensions": map[string]interface{}{
				"description": "List all extensions",
				"auth":        "required",
			},
			"POST /extensions": map[string]interface{}{
				"description": "Create new extension",
				"auth":        "required",
				"body": map[string]interface{}{
					"extension":          "string",
					"name":               "string",
					"password":           "string",
					"context":            "string (default: 'default')",
					"voicemail_enabled":  "boolean",
					"voicemail_password": "string",
				},
			},
			"PUT /extensions/:id": map[string]interface{}{
				"description": "Update extension",
				"auth":        "required",
			},
			"DELETE /extensions/:id": map[string]interface{}{
				"description": "Delete extension",
				"auth":        "required",
			},
		},
		"trunks": map[string]interface{}{
			"GET /trunks": map[string]interface{}{
				"description": "List all trunks",
				"auth":        "required",
			},
			"POST /trunks": map[string]interface{}{
				"description": "Create new trunk",
				"auth":        "required",
				"body": map[string]interface{}{
					"name":     "string",
					"provider": "string (callcentric, voip.ms, twilio, flowroute)",
					"host":     "string",
					"username": "string",
					"password": "string",
				},
			},
			"POST /trunks/:id/test": map[string]interface{}{
				"description": "Test trunk connectivity",
				"auth":        "required",
			},
		},
	}
}

// getWebSocketEvents returns WebSocket event documentation
func getWebSocketEvents() map[string]interface{} {
	return map[string]interface{}{
		"call_ringing": map[string]interface{}{
			"description": "New incoming call",
			"data": map[string]interface{}{
				"uuid":             "string",
				"caller_id_name":   "string",
				"caller_id_number": "string",
				"destination":      "string",
				"line":             "number",
			},
		},
		"call_answered": map[string]interface{}{
			"description": "Call was answered",
			"data": map[string]interface{}{
				"uuid":  "string",
				"state": "answered",
			},
		},
		"call_ended": map[string]interface{}{
			"description": "Call ended",
			"data": map[string]interface{}{
				"uuid":     "string",
				"duration": "number (seconds)",
			},
		},
		"call_update": map[string]interface{}{
			"description": "Call state changed",
			"data": map[string]interface{}{
				"uuid":  "string",
				"state": "string",
			},
		},
	}
}

// getAPIExamples returns code examples
func getAPIExamples() map[string]interface{} {
	return map[string]interface{}{
		"javascript": map[string]interface{}{
			"authentication": `// Login and get token
const response = await fetch('https://your-pbx.com/api/v1/auth/login', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({
    username: 'admin',
    password: 'your-password'
  })
});

const { token } = await response.json();

// Use token for subsequent requests
const calls = await fetch('https://your-pbx.com/api/v1/calls/active', {
  headers: { 'Authorization': 'Bearer ' + token }
}).then(r => r.json());`,
			
			"websocket": `// Connect to WebSocket
const ws = new WebSocket('wss://your-pbx.com/ws?token=' + token);

ws.onmessage = (event) => {
  const data = JSON.parse(event.data);
  
  switch(data.type) {
    case 'call_ringing':
      console.log('Incoming call from', data.data.caller_id_number);
      break;
    case 'call_answered':
      console.log('Call answered:', data.data.uuid);
      break;
  }
};`,
			
			"call_control": `// Answer a call
await fetch('https://your-pbx.com/api/v1/calls/' + callUUID + '/answer', {
  method: 'POST',
  headers: { 'Authorization': 'Bearer ' + token }
});

// Transfer to extension
await fetch('https://your-pbx.com/api/v1/calls/' + callUUID + '/transfer', {
  method: 'POST',
  headers: { 
    'Authorization': 'Bearer ' + token,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({ destination: '1001' })
});

// Add to conference
await fetch('https://your-pbx.com/api/v1/calls/' + callUUID + '/conference', {
  method: 'POST',
  headers: { 
    'Authorization': 'Bearer ' + token,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({ conference_id: 'morning-show' })
});`,
		},
		
		"curl": map[string]interface{}{
			"authentication": `# Login
curl -X POST https://your-pbx.com/api/v1/auth/login \
  -H "Content-Type: application/json" \
  -d '{"username":"admin","password":"your-password"}'

# Using API key instead
curl https://your-pbx.com/api/v1/calls/active \
  -H "X-API-Key: your-api-key"`,
			
			"call_control": `# Answer call
curl -X POST https://your-pbx.com/api/v1/calls/UUID/answer \
  -H "Authorization: Bearer TOKEN"

# Originate call
curl -X POST https://your-pbx.com/api/v1/calls/dial \
  -H "Authorization: Bearer TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"from":"+1234567890","to":"+0987654321"}'`,
		},
		
		"csharp": map[string]interface{}{
			"sdk_usage": `// Using AirwavePBX C# SDK
using AirwavePBX.SDK;

var client = new AirwavePBXClient("https://your-pbx.com", "your-api-key");

// Subscribe to events
client.OnCallRinging += (sender, e) => {
    Console.WriteLine($"Incoming: {e.Call.CallerIdNumber}");
};

// Control calls
await client.Calls.AnswerAsync(callUuid);
await client.Calls.TransferAsync(callUuid, "1001");
await client.Calls.ConferenceAsync(callUuid, "on-air");

// Get active calls
var activeCalls = await client.Calls.GetActiveAsync();`,
		},
	}
}