package api

import (
	"strings"
	
	"airwavepbx/internal/auth"
	"github.com/gofiber/fiber/v2"
)

// authMiddleware validates JWT tokens or API keys
func authMiddleware(authService *auth.Service) fiber.Handler {
	return func(c *fiber.Ctx) error {
		// Check for API key first
		apiKey := c.Get("X-API-Key")
		if apiKey != "" && authService.ValidateAPIKey(apiKey) {
			// Set a dummy claims object for API key auth
			c.Locals("claims", &auth.Claims{
				Username: "api",
				Admin:    true,
			})
			return c.Next()
		}
		
		// Check for Bearer token
		authHeader := c.Get("Authorization")
		if authHeader == "" {
			return c.Status(401).JSON(fiber.Map{"error": "Missing authorization"})
		}
		
		// Extract token
		parts := strings.Split(authHeader, " ")
		if len(parts) != 2 || parts[0] != "Bearer" {
			return c.Status(401).JSON(fiber.Map{"error": "Invalid authorization format"})
		}
		
		token := parts[1]
		
		// Validate token
		claims, err := authService.ValidateToken(token)
		if err != nil {
			return c.Status(401).JSON(fiber.Map{"error": "Invalid token"})
		}
		
		// Store claims in context
		c.Locals("claims", claims)
		
		return c.Next()
	}
}