package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"os/signal"
	"strings"
	"syscall"

	"airwavepbx/internal/api"
	"airwavepbx/internal/auth"
	"airwavepbx/internal/config"
	"airwavepbx/internal/database"
	"airwavepbx/internal/ami"
	"airwavepbx/internal/services"
	"airwavepbx/internal/websocket"
	"github.com/gofiber/fiber/v2"
	"github.com/gofiber/fiber/v2/middleware/compress"
	"github.com/gofiber/fiber/v2/middleware/cors"
	"github.com/gofiber/fiber/v2/middleware/logger"
	"github.com/gofiber/fiber/v2/middleware/recover"
)

var (
	Version = "1.0.8"
)

func main() {
	// Command line flags
	configFile := flag.String("config", "config.yaml", "Configuration file path")
	showVersion := flag.Bool("version", false, "Show version")
	flag.Parse()

	if *showVersion {
		fmt.Printf("AirwavePBX v%s\n", Version)
		os.Exit(0)
	}

	// Load configuration
	cfg, err := config.Load(*configFile)
	if err != nil {
		log.Fatalf("Failed to load configuration: %v", err)
	}

	// Initialize database
	db, err := database.Initialize(cfg.Database.Path)
	if err != nil {
		log.Fatalf("Failed to initialize database: %v", err)
	}
	defer db.Close()

	// Initialize authentication
	authService := auth.NewService(cfg.Admin.Username, cfg.Admin.Password, cfg.API.Key)

	// Initialize Asterisk AMI connection
	amiClient, err := ami.NewClient(cfg.Asterisk.AMIHost, cfg.Asterisk.AMIPort, cfg.Asterisk.AMIUsername, cfg.Asterisk.AMIPassword)
	if err != nil {
		log.Fatalf("Failed to connect to Asterisk AMI: %v", err)
	}
	defer amiClient.Close()

	// Initialize services
	callService := services.NewCallService(db, amiClient)
	extensionService := services.NewExtensionService(db, amiClient)
	trunkService := services.NewTrunkService(db, amiClient)
	
	// Initialize WebSocket hub
	wsHub := websocket.NewHub()
	go wsHub.Run()

	// Subscribe to Asterisk AMI events
	go amiClient.SubscribeEvents(wsHub)

	// Create Fiber app
	app := fiber.New(fiber.Config{
		AppName:      fmt.Sprintf("AirwavePBX v%s", Version),
		ErrorHandler: customErrorHandler,
	})

	// Global middleware
	app.Use(recover.New())
	app.Use(logger.New(logger.Config{
		Format: "[${time}] ${status} - ${latency} ${method} ${path}\n",
	}))
	app.Use(compress.New())
	app.Use(cors.New(cors.Config{
		AllowOrigins:     strings.Join(cfg.API.CORSOrigins, ","),
		AllowCredentials: true,
		AllowHeaders:     "Origin, Content-Type, Accept, Authorization, X-API-Key",
	}))

	// API routes
	apiGroup := app.Group("/api/v1")
	api.SetupRoutes(apiGroup, authService, callService, extensionService, trunkService, wsHub)

	// WebSocket endpoint
	app.Get("/ws", websocket.Handler(wsHub))

	// Serve static files first - more specific routes
	app.Static("/css", "./web/static/css")
	app.Static("/js", "./web/static/js")
	app.Static("/fap", "./web/static/fap")
	app.Static("/logos", "./web/static/logos")
	app.Static("/assets", "./web/static")
	
	// Serve index.html for root
	app.Get("/", func(c *fiber.Ctx) error {
		return c.SendFile("./web/static/index.html")
	})

	// Catch-all for SPA routing (must be last)
	app.Get("/*", func(c *fiber.Ctx) error {
		return c.SendFile("./web/static/index.html")
	})

	// Graceful shutdown
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, os.Interrupt, syscall.SIGTERM)

	go func() {
		<-sigChan
		log.Println("Shutting down AirwavePBX...")
		_ = app.Shutdown()
	}()

	// Start server
	addr := fmt.Sprintf("%s:%d", cfg.Server.Host, cfg.Server.Port)
	log.Printf("AirwavePBX v%s starting on %s", Version, addr)
	
	if cfg.Server.TLSEnabled && fileExists(cfg.Server.TLSCert) && fileExists(cfg.Server.TLSKey) {
		log.Fatal(app.ListenTLS(addr, cfg.Server.TLSCert, cfg.Server.TLSKey))
	} else {
		if cfg.Server.TLSEnabled {
			log.Println("TLS certificates not found, falling back to HTTP")
		}
		log.Fatal(app.Listen(addr))
	}
}

func customErrorHandler(c *fiber.Ctx, err error) error {
	code := fiber.StatusInternalServerError
	if e, ok := err.(*fiber.Error); ok {
		code = e.Code
	}

	return c.Status(code).JSON(fiber.Map{
		"error": err.Error(),
		"code":  code,
	})
}

func fileExists(path string) bool {
	_, err := os.Stat(path)
	return err == nil
}