// AirwavePBX API Client

class AirwaveAPI {
    constructor() {
        this.baseURL = '/api/v1';
        this.token = localStorage.getItem('airwave_token');
    }
    
    // Set authentication token
    setToken(token) {
        this.token = token;
        if (token) {
            localStorage.setItem('airwave_token', token);
        } else {
            localStorage.removeItem('airwave_token');
        }
    }
    
    // Make API request
    async request(method, endpoint, data = null) {
        const url = `${this.baseURL}${endpoint}`;
        const options = {
            method,
            headers: {
                'Content-Type': 'application/json'
            }
        };
        
        // Add auth header
        if (this.token) {
            options.headers['Authorization'] = `Bearer ${this.token}`;
        }
        
        // Add body for POST/PUT
        if (data && ['POST', 'PUT', 'PATCH'].includes(method)) {
            options.body = JSON.stringify(data);
        }
        
        try {
            const response = await fetch(url, options);
            
            // Handle auth errors
            if (response.status === 401) {
                this.setToken(null);
                window.location.href = '/';
                throw new Error('Authentication required');
            }
            
            const responseData = await response.json();
            
            if (!response.ok) {
                throw new Error(responseData.error || `HTTP ${response.status}`);
            }
            
            return responseData;
        } catch (error) {
            console.error('API Error:', error);
            throw error;
        }
    }
    
    // Convenience methods
    get(endpoint) {
        return this.request('GET', endpoint);
    }
    
    post(endpoint, data) {
        return this.request('POST', endpoint, data);
    }
    
    put(endpoint, data) {
        return this.request('PUT', endpoint, data);
    }
    
    delete(endpoint) {
        return this.request('DELETE', endpoint);
    }
    
    // Auth endpoints
    async login(username, password) {
        const response = await this.post('/auth/login', { username, password });
        if (response.token) {
            this.setToken(response.token);
        }
        return response;
    }
    
    async logout() {
        await this.post('/auth/logout');
        this.setToken(null);
    }
    
    getMe() {
        return this.get('/auth/me');
    }
    
    // Call endpoints
    getActiveCalls() {
        return this.get('/calls/active');
    }
    
    getCallHistory(limit = 50, offset = 0) {
        return this.get(`/calls/history?limit=${limit}&offset=${offset}`);
    }
    
    answerCall(uuid) {
        return this.post(`/calls/${uuid}/answer`);
    }
    
    hangupCall(uuid) {
        return this.post(`/calls/${uuid}/hangup`);
    }
    
    holdCall(uuid) {
        return this.post(`/calls/${uuid}/hold`);
    }
    
    unholdCall(uuid) {
        return this.post(`/calls/${uuid}/unhold`);
    }
    
    transferCall(uuid, destination) {
        return this.post(`/calls/${uuid}/transfer`, { destination });
    }
    
    conferenceCall(uuid, conferenceId = 'on-air') {
        return this.post(`/calls/${uuid}/conference`, { conference_id: conferenceId });
    }
    
    muteCall(uuid) {
        return this.post(`/calls/${uuid}/mute`);
    }
    
    unmuteCall(uuid) {
        return this.post(`/calls/${uuid}/unmute`);
    }
    
    setOnAir(uuid, onAir) {
        return this.post(`/calls/${uuid}/on-air`, { on_air: onAir });
    }
    
    dialCall(from, to) {
        return this.post('/calls/dial', { from, to });
    }
    
    getCallerInfo(uuid) {
        return this.get(`/calls/${uuid}/caller-info`);
    }
    
    // Extension endpoints
    getExtensions() {
        return this.get('/extensions');
    }
    
    getExtension(id) {
        return this.get(`/extensions/${id}`);
    }
    
    createExtension(data) {
        return this.post('/extensions', data);
    }
    
    updateExtension(id, data) {
        return this.put(`/extensions/${id}`, data);
    }
    
    deleteExtension(id) {
        return this.delete(`/extensions/${id}`);
    }
    
    // Trunk endpoints
    getTrunks() {
        return this.get('/trunks');
    }
    
    getTrunk(id) {
        return this.get(`/trunks/${id}`);
    }
    
    createTrunk(data) {
        return this.post('/trunks', data);
    }
    
    updateTrunk(id, data) {
        return this.put(`/trunks/${id}`, data);
    }
    
    deleteTrunk(id) {
        return this.delete(`/trunks/${id}`);
    }
    
    testTrunk(id) {
        return this.post(`/trunks/${id}/test`);
    }
    
    // System endpoints
    getSystemStatus() {
        return this.get('/system/status');
    }
    
    getSettings() {
        return this.get('/system/settings');
    }
    
    updateSettings(settings) {
        return this.put('/system/settings', settings);
    }
    
    getAPIDocs() {
        return this.get('/docs');
    }
}

// Create global API instance
window.API = new AirwaveAPI();