// Active Calls Page

async function loadCallsPage() {
    const contentDiv = document.getElementById('page-content');
    
    contentDiv.innerHTML = `
        <div class="page-header">
            <h1 class="page-title">Active Calls</h1>
            <div class="page-actions">
                <button class="btn btn-primary" onclick="showDialPad()">
                    <i class="fap fa-phone-plus"></i>
                    <span>New Call</span>
                </button>
            </div>
        </div>
        
        <div id="calls-container" class="calls-grid">
            <div class="loading">Loading calls...</div>
        </div>
    `;
    
    // Load calls
    await refreshActiveCalls();
    
    // Set up auto-refresh
    if (window.callsRefreshInterval) {
        clearInterval(window.callsRefreshInterval);
    }
    window.callsRefreshInterval = setInterval(refreshActiveCalls, 2000);
}

async function refreshActiveCalls() {
    if (App.currentPage !== 'calls') return;
    
    try {
        const calls = await API.getActiveCalls();
        const container = document.getElementById('calls-container');
        
        if (calls.length === 0) {
            container.innerHTML = `
                <div class="empty-state">
                    <i class="fap fa-phone-slash" style="font-size: 48px; color: var(--text-tertiary);"></i>
                    <p class="mt-3">No active calls</p>
                </div>
            `;
            return;
        }
        
        container.innerHTML = calls.map(call => renderCallCard(call)).join('');
    } catch (error) {
        console.error('Failed to refresh calls:', error);
    }
}

function renderCallCard(call) {
    const statusClass = call.state === 'ringing' ? 'ringing' : 'active';
    const duration = call.answer_time ? 
        Math.floor((new Date() - new Date(call.answer_time)) / 1000) : 0;
    
    return `
        <div class="call-card glass-card ${statusClass}" id="call-${call.uuid}">
            <div class="call-header">
                <div class="call-status">
                    <div class="status-indicator ${statusClass}"></div>
                    <span>${call.state.charAt(0).toUpperCase() + call.state.slice(1)}</span>
                </div>
                <div class="call-line">
                    Line ${call.line_number || 1}
                </div>
            </div>
            
            <div class="call-info">
                <div class="call-number">${formatPhoneNumber(call.caller_id_number)}</div>
                <div class="call-name">${call.caller_id_name || 'Unknown Caller'}</div>
                ${call.state !== 'ringing' ? `
                    <div class="call-duration" data-start="${call.answer_time || call.start_time}">
                        ${formatDuration(duration)}
                    </div>
                ` : ''}
            </div>
            
            <div class="call-actions">
                ${renderCallActions(call)}
            </div>
            
            ${call.conference_id ? `
                <div class="call-conference">
                    <i class="fap fa-users"></i>
                    <span>Conference: ${call.conference_id}</span>
                </div>
            ` : ''}
        </div>
    `;
}

function renderCallActions(call) {
    const actions = [];
    
    if (call.state === 'ringing') {
        actions.push(`
            <button class="btn btn-success" onclick="answerCall('${call.uuid}')">
                <i class="fap fa-phone-alt"></i>
                <span>Answer</span>
            </button>
        `);
    }
    
    if (call.state === 'active' || call.state === 'answered') {
        if (!call.is_on_air) {
            actions.push(`
                <button class="btn btn-primary" onclick="setOnAir('${call.uuid}', true)">
                    <i class="fap fa-broadcast-tower"></i>
                    <span>On Air</span>
                </button>
            `);
        } else {
            actions.push(`
                <button class="btn btn-warning" onclick="setOnAir('${call.uuid}', false)">
                    <i class="fap fa-broadcast-tower"></i>
                    <span>Off Air</span>
                </button>
            `);
        }
        
        if (!call.is_muted) {
            actions.push(`
                <button class="btn" onclick="muteCall('${call.uuid}')">
                    <i class="fap fa-microphone-slash"></i>
                    <span>Mute</span>
                </button>
            `);
        } else {
            actions.push(`
                <button class="btn btn-warning" onclick="unmuteCall('${call.uuid}')">
                    <i class="fap fa-microphone"></i>
                    <span>Unmute</span>
                </button>
            `);
        }
        
        actions.push(`
            <button class="btn" onclick="showTransferDialog('${call.uuid}')">
                <i class="fap fa-exchange-alt"></i>
                <span>Transfer</span>
            </button>
        `);
    }
    
    actions.push(`
        <button class="btn btn-danger" onclick="hangupCall('${call.uuid}')">
            <i class="fap fa-phone-slash"></i>
            <span>Hangup</span>
        </button>
    `);
    
    return actions.join('');
}

// Call control functions
async function answerCall(uuid) {
    try {
        await API.answerCall(uuid);
        showNotification('Call Answered', 'Call has been answered', 'success');
        refreshActiveCalls();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

async function hangupCall(uuid) {
    try {
        await API.hangupCall(uuid);
        showNotification('Call Ended', 'Call has been terminated', 'info');
        refreshActiveCalls();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

async function muteCall(uuid) {
    try {
        await API.muteCall(uuid);
        refreshActiveCalls();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

async function unmuteCall(uuid) {
    try {
        await API.unmuteCall(uuid);
        refreshActiveCalls();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

async function setOnAir(uuid, onAir) {
    try {
        await API.setOnAir(uuid, onAir);
        showNotification(
            onAir ? 'On Air' : 'Off Air', 
            onAir ? 'Call is now on air' : 'Call is off air', 
            onAir ? 'success' : 'info'
        );
        refreshActiveCalls();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

function showTransferDialog(uuid) {
    showModal('Transfer Call', `
        <form id="transfer-form" onsubmit="handleTransfer(event, '${uuid}')">
            <div class="form-group">
                <label>Transfer to Extension:</label>
                <div class="glass-input">
                    <i class="fap fa-phone input-icon"></i>
                    <input type="text" name="destination" placeholder="Extension number" required>
                </div>
            </div>
        </form>
    `, [
        {
            label: 'Cancel',
            onclick: 'closeModal()'
        },
        {
            label: 'Transfer',
            class: 'btn-primary',
            icon: 'exchange-alt',
            onclick: "document.getElementById('transfer-form').submit()"
        }
    ]);
}

async function handleTransfer(event, uuid) {
    event.preventDefault();
    const formData = new FormData(event.target);
    const destination = formData.get('destination');
    
    try {
        await API.transferCall(uuid, destination);
        closeModal();
        showNotification('Call Transferred', `Call transferred to ${destination}`, 'success');
        refreshActiveCalls();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

function showDialPad() {
    showModal('New Call', `
        <form id="dial-form" onsubmit="handleDial(event)">
            <div class="form-group">
                <label>From (Caller ID):</label>
                <div class="glass-input">
                    <i class="fap fa-user input-icon"></i>
                    <input type="tel" name="from" placeholder="Your phone number" required>
                </div>
            </div>
            <div class="form-group">
                <label>To (Destination):</label>
                <div class="glass-input">
                    <i class="fap fa-phone input-icon"></i>
                    <input type="tel" name="to" placeholder="Number to call" required>
                </div>
            </div>
        </form>
    `, [
        {
            label: 'Cancel',
            onclick: 'closeModal()'
        },
        {
            label: 'Call',
            class: 'btn-primary',
            icon: 'phone-alt',
            onclick: "document.getElementById('dial-form').submit()"
        }
    ]);
}

async function handleDial(event) {
    event.preventDefault();
    const formData = new FormData(event.target);
    const from = formData.get('from');
    const to = formData.get('to');
    
    try {
        const response = await API.dialCall(from, to);
        closeModal();
        showNotification('Call Initiated', `Calling ${to}...`, 'success');
        refreshActiveCalls();
    } catch (error) {
        showNotification('Error', error.message, 'error');
    }
}

// Update call durations
setInterval(() => {
    if (App.currentPage === 'calls') {
        document.querySelectorAll('.call-duration').forEach(el => {
            const start = new Date(el.dataset.start);
            const duration = Math.floor((new Date() - start) / 1000);
            el.textContent = formatDuration(duration);
        });
    }
}, 1000);

// Clean up interval on page change
window.addEventListener('beforeunload', () => {
    if (window.callsRefreshInterval) {
        clearInterval(window.callsRefreshInterval);
    }
});

// Add calls-specific styles
const callsStyles = `
<style>
.empty-state {
    text-align: center;
    padding: var(--spacing-xl);
    color: var(--text-secondary);
}

.call-conference {
    display: flex;
    align-items: center;
    gap: var(--spacing-xs);
    margin-top: var(--spacing-md);
    padding-top: var(--spacing-md);
    border-top: 1px solid var(--glass-border);
    font-size: 14px;
    color: var(--text-secondary);
}

.call-line {
    background: var(--glass-bg);
    padding: var(--spacing-xs) var(--spacing-sm);
    border-radius: var(--radius-sm);
    font-size: 12px;
    font-weight: 600;
}

.call-duration {
    font-size: 14px;
    color: var(--text-secondary);
    margin-top: var(--spacing-xs);
}
</style>
`;

if (!document.getElementById('calls-styles')) {
    const styleElement = document.createElement('div');
    styleElement.id = 'calls-styles';
    styleElement.innerHTML = callsStyles;
    document.head.appendChild(styleElement);
}