package api

import (
	"strconv"
	"time"
	
	"airwavepbx/internal/auth"
	"airwavepbx/internal/services"
	"github.com/gofiber/fiber/v2"
)

// Auth handlers

func handleLogin(authService *auth.Service) fiber.Handler {
	return func(c *fiber.Ctx) error {
		var req struct {
			Username string `json:"username"`
			Password string `json:"password"`
		}
		
		if err := c.BodyParser(&req); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		if err := authService.ValidateCredentials(req.Username, req.Password); err != nil {
			return c.Status(401).JSON(fiber.Map{"error": "Invalid credentials"})
		}
		
		token, err := authService.GenerateToken(req.Username)
		if err != nil {
			return c.Status(500).JSON(fiber.Map{"error": "Failed to generate token"})
		}
		
		return c.JSON(fiber.Map{
			"token": token,
			"user": fiber.Map{
				"username": req.Username,
				"admin":    true,
			},
		})
	}
}

func handleLogout() fiber.Handler {
	return func(c *fiber.Ctx) error {
		// Token invalidation would be handled here if using a blacklist
		return c.JSON(fiber.Map{"message": "Logged out successfully"})
	}
}

func handleGetCurrentUser() fiber.Handler {
	return func(c *fiber.Ctx) error {
		claims := c.Locals("claims").(*auth.Claims)
		return c.JSON(fiber.Map{
			"username": claims.Username,
			"admin":    claims.Admin,
		})
	}
}

// Call handlers

func handleGetActiveCalls(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		calls, err := callService.GetActiveCalls()
		if err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(calls)
	}
}

func handleGetCallHistory(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		limit := c.QueryInt("limit", 50)
		offset := c.QueryInt("offset", 0)
		
		calls, total, err := callService.GetCallHistory(limit, offset)
		if err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(fiber.Map{
			"calls": calls,
			"total": total,
			"limit": limit,
			"offset": offset,
		})
	}
}

func handleAnswerCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		if err := callService.AnswerCall(uuid); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(fiber.Map{"message": "Call answered"})
	}
}

func handleHangupCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		if err := callService.HangupCall(uuid); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(fiber.Map{"message": "Call ended"})
	}
}

func handleHoldCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		if err := callService.HoldCall(uuid); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(fiber.Map{"message": "Call held"})
	}
}

func handleUnholdCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		if err := callService.UnholdCall(uuid); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(fiber.Map{"message": "Call resumed"})
	}
}

func handleTransferCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		var req struct {
			Destination string `json:"destination"`
		}
		
		if err := c.BodyParser(&req); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		if err := callService.TransferCall(uuid, req.Destination); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(fiber.Map{"message": "Call transferred"})
	}
}

func handleConferenceCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		var req struct {
			ConferenceID string `json:"conference_id"`
		}
		
		if err := c.BodyParser(&req); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		if req.ConferenceID == "" {
			req.ConferenceID = "on-air"
		}
		
		if err := callService.ConferenceCall(uuid, req.ConferenceID); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(fiber.Map{"message": "Call added to conference"})
	}
}

func handleMuteCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		if err := callService.MuteCall(uuid); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(fiber.Map{"message": "Call muted"})
	}
}

func handleUnmuteCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		if err := callService.UnmuteCall(uuid); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(fiber.Map{"message": "Call unmuted"})
	}
}

func handleSetOnAir(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		uuid := c.Params("uuid")
		var req struct {
			OnAir bool `json:"on_air"`
		}
		
		if err := c.BodyParser(&req); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		if err := callService.SetOnAir(uuid, req.OnAir); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(fiber.Map{"message": "On-air status updated"})
	}
}

func handleDialCall(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		var req struct {
			From string `json:"from"`
			To   string `json:"to"`
		}
		
		if err := c.BodyParser(&req); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		uuid, err := callService.CreateOutboundCall(req.From, req.To)
		if err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(fiber.Map{
			"uuid": uuid,
			"message": "Call initiated",
		})
	}
}

func handleGetCallerInfo(callService *services.CallService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		phoneNumber := c.Query("phone", "")
		if phoneNumber == "" {
			// Try to get from active call
			uuid := c.Params("uuid")
			calls, err := callService.GetActiveCalls()
			if err != nil {
				return c.Status(500).JSON(fiber.Map{"error": err.Error()})
			}
			
			for _, call := range calls {
				if call.UUID == uuid {
					phoneNumber = call.CallerIDNumber
					break
				}
			}
		}
		
		if phoneNumber == "" {
			return c.Status(400).JSON(fiber.Map{"error": "Phone number required"})
		}
		
		caller, err := callService.GetCallerInfo(phoneNumber)
		if err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(caller)
	}
}

// Extension handlers

func handleGetExtensions(extensionService *services.ExtensionService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		extensions, err := extensionService.GetAll()
		if err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(extensions)
	}
}

func handleGetExtension(extensionService *services.ExtensionService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		id, err := strconv.Atoi(c.Params("id"))
		if err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid ID"})
		}
		
		extension, err := extensionService.Get(id)
		if err != nil {
			return c.Status(404).JSON(fiber.Map{"error": "Extension not found"})
		}
		
		return c.JSON(extension)
	}
}

func handleCreateExtension(extensionService *services.ExtensionService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		var extension services.Extension
		if err := c.BodyParser(&extension); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		if err := extensionService.Create(&extension); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.Status(201).JSON(extension)
	}
}

func handleUpdateExtension(extensionService *services.ExtensionService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		id, err := strconv.Atoi(c.Params("id"))
		if err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid ID"})
		}
		
		var extension services.Extension
		if err := c.BodyParser(&extension); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		extension.ID = id
		if err := extensionService.Update(&extension); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(extension)
	}
}

func handleDeleteExtension(extensionService *services.ExtensionService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		id, err := strconv.Atoi(c.Params("id"))
		if err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid ID"})
		}
		
		if err := extensionService.Delete(id); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(fiber.Map{"message": "Extension deleted"})
	}
}

// Trunk handlers

func handleGetTrunks(trunkService *services.TrunkService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		trunks, err := trunkService.GetAll()
		if err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		return c.JSON(trunks)
	}
}

func handleGetTrunk(trunkService *services.TrunkService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		id, err := strconv.Atoi(c.Params("id"))
		if err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid ID"})
		}
		
		trunk, err := trunkService.Get(id)
		if err != nil {
			return c.Status(404).JSON(fiber.Map{"error": "Trunk not found"})
		}
		
		return c.JSON(trunk)
	}
}

func handleCreateTrunk(trunkService *services.TrunkService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		var trunk services.Trunk
		if err := c.BodyParser(&trunk); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		if err := trunkService.Create(&trunk); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.Status(201).JSON(trunk)
	}
}

func handleUpdateTrunk(trunkService *services.TrunkService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		id, err := strconv.Atoi(c.Params("id"))
		if err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid ID"})
		}
		
		var trunk services.Trunk
		if err := c.BodyParser(&trunk); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		trunk.ID = id
		if err := trunkService.Update(&trunk); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(trunk)
	}
}

func handleDeleteTrunk(trunkService *services.TrunkService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		id, err := strconv.Atoi(c.Params("id"))
		if err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid ID"})
		}
		
		if err := trunkService.Delete(id); err != nil {
			return c.Status(500).JSON(fiber.Map{"error": err.Error()})
		}
		
		return c.JSON(fiber.Map{"message": "Trunk deleted"})
	}
}

func handleTestTrunk(trunkService *services.TrunkService) fiber.Handler {
	return func(c *fiber.Ctx) error {
		id, err := strconv.Atoi(c.Params("id"))
		if err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid ID"})
		}
		
		if err := trunkService.TestTrunk(id); err != nil {
			return c.Status(500).JSON(fiber.Map{
				"status": "failed",
				"error": err.Error(),
			})
		}
		
		return c.JSON(fiber.Map{
			"status": "success",
			"message": "Trunk is working",
		})
	}
}

// System handlers

func handleGetSystemStatus() fiber.Handler {
	return func(c *fiber.Ctx) error {
		return c.JSON(fiber.Map{
			"status": "online",
			"version": "1.0.2",
			"uptime": time.Since(startTime).Seconds(),
			"services": fiber.Map{
				"api": "running",
				"asterisk": "running",
				"database": "connected",
			},
		})
	}
}

func handleGetSettings() fiber.Handler {
	return func(c *fiber.Ctx) error {
		// TODO: Implement settings retrieval from database
		return c.JSON(fiber.Map{
			"company_name": "AirwavePBX",
			"timezone": "UTC",
			"recording_enabled": true,
			"recording_format": "wav",
			"max_call_duration": 3600,
		})
	}
}

func handleUpdateSettings() fiber.Handler {
	return func(c *fiber.Ctx) error {
		var settings map[string]interface{}
		if err := c.BodyParser(&settings); err != nil {
			return c.Status(400).JSON(fiber.Map{"error": "Invalid request"})
		}
		
		// TODO: Implement settings update in database
		
		return c.JSON(fiber.Map{"message": "Settings updated"})
	}
}

func handleGetAPIDocs() fiber.Handler {
	return func(c *fiber.Ctx) error {
		return c.JSON(fiber.Map{
			"version": "1.0.2",
			"title": "AirwavePBX API Documentation",
			"description": "Complete API reference for AirwavePBX broadcasting phone system",
			"base_url": "https://your-domain.com/api/v1",
			"authentication": fiber.Map{
				"type": "Bearer token or API key",
				"headers": fiber.Map{
					"Authorization": "Bearer <token>",
					"X-API-Key": "<api-key>",
				},
			},
			"endpoints": getAPIEndpoints(),
			"websocket": fiber.Map{
				"url": "wss://your-domain.com/ws",
				"authentication": "Include token as query parameter: ?token=<token>",
				"events": getWebSocketEvents(),
			},
			"examples": getAPIExamples(),
		})
	}
}

var startTime = time.Now()