package database

import (
	"database/sql"
	"fmt"
	"os"
	"path/filepath"
	
	_ "github.com/mattn/go-sqlite3"
)

// DB wraps the database connection
type DB struct {
	*sql.DB
}

// Initialize creates and sets up the database
func Initialize(dbPath string) (*DB, error) {
	// Ensure directory exists
	dir := filepath.Dir(dbPath)
	if err := os.MkdirAll(dir, 0755); err != nil {
		return nil, fmt.Errorf("failed to create database directory: %w", err)
	}
	
	// Open database
	sqlDB, err := sql.Open("sqlite3", dbPath+"?_foreign_keys=on")
	if err != nil {
		return nil, fmt.Errorf("failed to open database: %w", err)
	}
	
	// Test connection
	if err := sqlDB.Ping(); err != nil {
		return nil, fmt.Errorf("failed to connect to database: %w", err)
	}
	
	db := &DB{sqlDB}
	
	// Run migrations
	if err := db.migrate(); err != nil {
		return nil, fmt.Errorf("failed to migrate database: %w", err)
	}
	
	return db, nil
}

// migrate runs database migrations
func (db *DB) migrate() error {
	migrations := []string{
		// Extensions table
		`CREATE TABLE IF NOT EXISTS extensions (
			id INTEGER PRIMARY KEY AUTOINCREMENT,
			extension VARCHAR(10) UNIQUE NOT NULL,
			name VARCHAR(100) NOT NULL,
			password VARCHAR(100) NOT NULL,
			context VARCHAR(50) DEFAULT 'default',
			callerid_name VARCHAR(100),
			callerid_number VARCHAR(20),
			voicemail_enabled BOOLEAN DEFAULT 0,
			voicemail_password VARCHAR(10),
			enabled BOOLEAN DEFAULT 1,
			created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
		)`,
		
		// Trunks table
		`CREATE TABLE IF NOT EXISTS trunks (
			id INTEGER PRIMARY KEY AUTOINCREMENT,
			name VARCHAR(100) UNIQUE NOT NULL,
			provider VARCHAR(50) NOT NULL,
			host VARCHAR(255) NOT NULL,
			username VARCHAR(100),
			password VARCHAR(100),
			from_user VARCHAR(100),
			from_domain VARCHAR(255),
			register BOOLEAN DEFAULT 1,
			codec_prefs VARCHAR(255) DEFAULT 'OPUS,G722,PCMU,PCMA',
			enabled BOOLEAN DEFAULT 1,
			created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
		)`,
		
		// Call logs table
		`CREATE TABLE IF NOT EXISTS call_logs (
			id INTEGER PRIMARY KEY AUTOINCREMENT,
			uuid VARCHAR(36) UNIQUE NOT NULL,
			direction VARCHAR(10) NOT NULL,
			caller_id_name VARCHAR(100),
			caller_id_number VARCHAR(20),
			destination_number VARCHAR(20),
			start_time DATETIME NOT NULL,
			answer_time DATETIME,
			end_time DATETIME,
			duration INTEGER DEFAULT 0,
			billsec INTEGER DEFAULT 0,
			disposition VARCHAR(20),
			recording_path VARCHAR(255),
			trunk_id INTEGER,
			extension_id INTEGER,
			created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
			FOREIGN KEY (trunk_id) REFERENCES trunks(id),
			FOREIGN KEY (extension_id) REFERENCES extensions(id)
		)`,
		
		// Callers table for broadcaster integration
		`CREATE TABLE IF NOT EXISTS callers (
			id INTEGER PRIMARY KEY AUTOINCREMENT,
			phone_number VARCHAR(20) UNIQUE NOT NULL,
			first_name VARCHAR(100),
			last_name VARCHAR(100),
			city VARCHAR(100),
			state VARCHAR(50),
			country VARCHAR(50),
			last_called DATETIME,
			call_count INTEGER DEFAULT 0,
			notes TEXT,
			tags TEXT,
			banned BOOLEAN DEFAULT 0,
			vip BOOLEAN DEFAULT 0,
			created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
		)`,
		
		// Active calls table for real-time tracking
		`CREATE TABLE IF NOT EXISTS active_calls (
			uuid VARCHAR(36) PRIMARY KEY,
			caller_id_name VARCHAR(100),
			caller_id_number VARCHAR(20),
			destination_number VARCHAR(20),
			line_number INTEGER,
			state VARCHAR(20),
			answered_by VARCHAR(100),
			start_time DATETIME NOT NULL,
			answer_time DATETIME,
			conference_id VARCHAR(50),
			is_muted BOOLEAN DEFAULT 0,
			is_on_air BOOLEAN DEFAULT 0,
			created_at DATETIME DEFAULT CURRENT_TIMESTAMP
		)`,
		
		// API keys table
		`CREATE TABLE IF NOT EXISTS api_keys (
			id INTEGER PRIMARY KEY AUTOINCREMENT,
			name VARCHAR(100) NOT NULL,
			key_hash VARCHAR(255) UNIQUE NOT NULL,
			permissions TEXT,
			last_used DATETIME,
			expires_at DATETIME,
			created_at DATETIME DEFAULT CURRENT_TIMESTAMP
		)`,
		
		// System settings table
		`CREATE TABLE IF NOT EXISTS settings (
			key VARCHAR(100) PRIMARY KEY,
			value TEXT,
			updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
		)`,
		
		// Create indexes
		`CREATE INDEX IF NOT EXISTS idx_call_logs_start_time ON call_logs(start_time)`,
		`CREATE INDEX IF NOT EXISTS idx_call_logs_caller_number ON call_logs(caller_id_number)`,
		`CREATE INDEX IF NOT EXISTS idx_callers_phone ON callers(phone_number)`,
		`CREATE INDEX IF NOT EXISTS idx_active_calls_state ON active_calls(state)`,
		
		// Create triggers for updated_at
		`CREATE TRIGGER IF NOT EXISTS update_extensions_timestamp 
		AFTER UPDATE ON extensions
		BEGIN
			UPDATE extensions SET updated_at = CURRENT_TIMESTAMP WHERE id = NEW.id;
		END`,
		
		`CREATE TRIGGER IF NOT EXISTS update_trunks_timestamp 
		AFTER UPDATE ON trunks
		BEGIN
			UPDATE trunks SET updated_at = CURRENT_TIMESTAMP WHERE id = NEW.id;
		END`,
		
		`CREATE TRIGGER IF NOT EXISTS update_callers_timestamp 
		AFTER UPDATE ON callers
		BEGIN
			UPDATE callers SET updated_at = CURRENT_TIMESTAMP WHERE id = NEW.id;
		END`,
		
		// Insert default settings
		`INSERT OR IGNORE INTO settings (key, value) VALUES 
			('company_name', 'AirwavePBX'),
			('timezone', 'UTC'),
			('recording_enabled', 'true'),
			('recording_format', 'wav'),
			('max_call_duration', '3600')`,
	}
	
	// Execute migrations
	for _, migration := range migrations {
		if _, err := db.Exec(migration); err != nil {
			return fmt.Errorf("migration failed: %w\nSQL: %s", err, migration)
		}
	}
	
	return nil
}