// FAP Icons Loader (FontAwesome Pro replacement)
// Loads duotone SVG icons from local assets

(function() {
    'use strict';
    
    // Icon cache
    const iconCache = new Map();
    
    // Load SVG icon
    async function loadIcon(iconName, options = {}) {
        const cacheKey = `${iconName}-${JSON.stringify(options)}`;
        
        if (iconCache.has(cacheKey)) {
            return iconCache.get(cacheKey);
        }
        
        try {
            const response = await fetch(`/fap/svgs/duotone/${iconName}.svg`);
            if (!response.ok) throw new Error(`Icon not found: ${iconName}`);
            
            let svg = await response.text();
            
            // Apply color theming if specified
            if (options.primaryColor || options.secondaryColor) {
                svg = applyIconTheming(svg, options);
            }
            
            iconCache.set(cacheKey, svg);
            return svg;
        } catch (error) {
            console.warn(`Failed to load icon: ${iconName}`, error);
            return getPlaceholderIcon();
        }
    }
    
    // Apply duotone color theming
    function applyIconTheming(svg, options) {
        const parser = new DOMParser();
        const doc = parser.parseFromString(svg, 'image/svg+xml');
        const svgElement = doc.querySelector('svg');
        
        if (!svgElement) return svg;
        
        // Find primary and secondary paths
        const paths = svgElement.querySelectorAll('path');
        if (paths.length >= 2) {
            // Primary layer (usually the second path)
            if (options.primaryColor && paths[1]) {
                paths[1].setAttribute('fill', options.primaryColor);
                paths[1].setAttribute('opacity', options.primaryOpacity || '1');
            }
            
            // Secondary layer (usually the first path)
            if (options.secondaryColor && paths[0]) {
                paths[0].setAttribute('fill', options.secondaryColor);
                paths[0].setAttribute('opacity', options.secondaryOpacity || '0.4');
            }
        }
        
        return new XMLSerializer().serializeToString(doc);
    }
    
    // Get placeholder icon for missing icons
    function getPlaceholderIcon() {
        return `<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
            <path opacity="0.4" d="M256 512c141.4 0 256-114.6 256-256S397.4 0 256 0 0 114.6 0 256s114.6 256 256 256z"/>
            <path d="M256 128c17.7 0 32 14.3 32 32v128c0 17.7-14.3 32-32 32s-32-14.3-32-32V160c0-17.7 14.3-32 32-32zm32 256c0 17.7-14.3 32-32 32s-32-14.3-32-32 14.3-32 32-32 32 14.3 32 32z"/>
        </svg>`;
    }
    
    // Process all FAP icons on the page
    async function processIcons() {
        const icons = document.querySelectorAll('.fap:not(.fap-processed)');
        
        for (const icon of icons) {
            icon.classList.add('fap-processed');
            
            // Extract icon name from classes
            const classes = Array.from(icon.classList);
            const iconClass = classes.find(c => c.startsWith('fa-'));
            
            if (!iconClass) continue;
            
            const iconName = iconClass.substring(3); // Remove 'fa-' prefix
            
            // Get theming options from data attributes or classes
            const options = {
                primaryColor: icon.dataset.primaryColor || getComputedStyle(icon).color,
                secondaryColor: icon.dataset.secondaryColor,
                primaryOpacity: icon.dataset.primaryOpacity,
                secondaryOpacity: icon.dataset.secondaryOpacity
            };
            
            // Load and insert icon
            const svg = await loadIcon(iconName, options);
            icon.innerHTML = svg;
            
            // Apply size classes
            const sizeClass = classes.find(c => c.match(/^fa-([0-9]+x|xs|sm|lg)$/));
            if (sizeClass) {
                icon.style.fontSize = getFontSize(sizeClass);
            }
            
            // Apply animation classes
            if (icon.classList.contains('fa-spin')) {
                icon.style.animation = 'fa-spin 2s linear infinite';
            }
            if (icon.classList.contains('fa-pulse')) {
                icon.style.animation = 'fa-pulse 2s ease-in-out infinite';
            }
        }
    }
    
    // Get font size for size classes
    function getFontSize(sizeClass) {
        const sizes = {
            'fa-xs': '0.75em',
            'fa-sm': '0.875em',
            'fa-lg': '1.33em',
            'fa-2x': '2em',
            'fa-3x': '3em',
            'fa-4x': '4em',
            'fa-5x': '5em'
        };
        return sizes[sizeClass] || '1em';
    }
    
    // Add CSS for FAP icons
    const style = document.createElement('style');
    style.textContent = `
        .fap {
            display: inline-block;
            width: 1em;
            height: 1em;
            vertical-align: -0.125em;
        }
        
        .fap svg {
            width: 100%;
            height: 100%;
        }
        
        @keyframes fa-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        @keyframes fa-pulse {
            0% { opacity: 1; }
            50% { opacity: 0.4; }
            100% { opacity: 1; }
        }
    `;
    document.head.appendChild(style);
    
    // Process icons on DOM changes
    const observer = new MutationObserver(() => {
        processIcons();
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
    
    // Initial processing
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', processIcons);
    } else {
        processIcons();
    }
    
    // Expose API for dynamic icon creation
    window.FAP = {
        loadIcon,
        processIcons,
        createIcon(iconName, options = {}) {
            const span = document.createElement('span');
            span.className = `fap fa-${iconName}`;
            Object.assign(span.dataset, options);
            return span;
        }
    };
})();