package ami

import (
	"bufio"
	"fmt"
	"log"
	"net"
	"strings"
	"sync"
	"time"
)

// Client represents an Asterisk Manager Interface connection
type Client struct {
	host     string
	port     int
	username string
	password string
	conn     net.Conn
	reader   *bufio.Reader
	writer   *bufio.Writer
	mu       sync.Mutex
	events   chan Event
	closed   bool
}

// Event represents an AMI event
type Event struct {
	Name    string
	Headers map[string]string
}

// Action represents an AMI action
type Action struct {
	Name    string
	Headers map[string]string
}

// NewClient creates a new AMI client
func NewClient(host string, port int, username, password string) (*Client, error) {
	client := &Client{
		host:     host,
		port:     port,
		username: username,
		password: password,
		events:   make(chan Event, 1000),
	}
	
	if err := client.connect(); err != nil {
		return nil, err
	}
	
	// Start event reader
	go client.readEvents()
	
	return client, nil
}

// connect establishes connection to Asterisk AMI
func (c *Client) connect() error {
	addr := fmt.Sprintf("%s:%d", c.host, c.port)
	conn, err := net.DialTimeout("tcp", addr, 10*time.Second)
	if err != nil {
		return fmt.Errorf("failed to connect to AMI: %w", err)
	}
	
	c.conn = conn
	c.reader = bufio.NewReader(conn)
	c.writer = bufio.NewWriter(conn)
	
	// Read welcome message
	line, err := c.reader.ReadString('\n')
	if err != nil {
		return fmt.Errorf("failed to read AMI welcome: %w", err)
	}
	
	if !strings.Contains(line, "Asterisk Call Manager") {
		return fmt.Errorf("unexpected AMI response: %s", line)
	}
	
	// Login
	loginAction := Action{
		Name: "Login",
		Headers: map[string]string{
			"Username": c.username,
			"Secret":   c.password,
		},
	}
	
	response, err := c.SendAction(loginAction)
	if err != nil {
		return fmt.Errorf("login failed: %w", err)
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("login failed: %s", response.Headers["Message"])
	}
	
	log.Println("Connected to Asterisk AMI")
	return nil
}

// SendAction sends an action to AMI and returns the response
func (c *Client) SendAction(action Action) (*Event, error) {
	c.mu.Lock()
	defer c.mu.Unlock()
	
	// Write action
	fmt.Fprintf(c.writer, "Action: %s\r\n", action.Name)
	for key, value := range action.Headers {
		fmt.Fprintf(c.writer, "%s: %s\r\n", key, value)
	}
	fmt.Fprintf(c.writer, "\r\n")
	
	if err := c.writer.Flush(); err != nil {
		return nil, err
	}
	
	// Read response
	return c.readResponse()
}

// readResponse reads a single response/event from AMI
func (c *Client) readResponse() (*Event, error) {
	event := &Event{
		Headers: make(map[string]string),
	}
	
	for {
		line, err := c.reader.ReadString('\n')
		if err != nil {
			return nil, err
		}
		
		line = strings.TrimSpace(line)
		if line == "" {
			break
		}
		
		parts := strings.SplitN(line, ":", 2)
		if len(parts) == 2 {
			key := strings.TrimSpace(parts[0])
			value := strings.TrimSpace(parts[1])
			event.Headers[key] = value
			
			if key == "Event" {
				event.Name = value
			}
		}
	}
	
	return event, nil
}

// readEvents continuously reads events from AMI
func (c *Client) readEvents() {
	for !c.closed {
		event, err := c.readResponse()
		if err != nil {
			if !c.closed {
				log.Printf("Error reading AMI event: %v", err)
				// Try to reconnect
				time.Sleep(5 * time.Second)
				if err := c.connect(); err != nil {
					log.Printf("Failed to reconnect: %v", err)
				}
			}
			continue
		}
		
		if event.Name != "" {
			select {
			case c.events <- *event:
			default:
				log.Println("Event channel full, dropping event")
			}
		}
	}
}

// SubscribeEvents handles incoming events and broadcasts them
func (c *Client) SubscribeEvents(hub interface{ BroadcastEvent(Event) }) {
	for event := range c.events {
		// Filter relevant events for broadcasting
		switch event.Name {
		case "Newchannel", "Newstate", "Hangup", "Bridge", "Unbridge", "DTMFBegin", "DTMFEnd":
			hub.BroadcastEvent(event)
		case "UserEvent":
			// Handle custom AirwavePBX events
			if strings.HasPrefix(event.Headers["UserEvent"], "AIRWAVE_") {
				hub.BroadcastEvent(event)
			}
		}
	}
}

// Originate makes an outbound call
func (c *Client) Originate(from, to, context string) error {
	action := Action{
		Name: "Originate",
		Headers: map[string]string{
			"Channel":  fmt.Sprintf("PJSIP/%s", to),
			"Context":  context,
			"Exten":    to,
			"Priority": "1",
			"CallerID": from,
			"Timeout":  "30000",
		},
	}
	
	response, err := c.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("originate failed: %s", response.Headers["Message"])
	}
	
	return nil
}

// HangupCall hangs up a call by channel
func (c *Client) HangupCall(channel string) error {
	action := Action{
		Name: "Hangup",
		Headers: map[string]string{
			"Channel": channel,
		},
	}
	
	response, err := c.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("hangup failed: %s", response.Headers["Message"])
	}
	
	return nil
}

// TransferCall transfers a call
func (c *Client) TransferCall(channel, destination string) error {
	action := Action{
		Name: "Redirect",
		Headers: map[string]string{
			"Channel": channel,
			"Context": "internal",
			"Exten":   destination,
			"Priority": "1",
		},
	}
	
	response, err := c.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("transfer failed: %s", response.Headers["Message"])
	}
	
	return nil
}

// ParkCall parks a call
func (c *Client) ParkCall(channel string) (string, error) {
	action := Action{
		Name: "Park",
		Headers: map[string]string{
			"Channel": channel,
		},
	}
	
	response, err := c.SendAction(action)
	if err != nil {
		return "", err
	}
	
	if response.Headers["Response"] != "Success" {
		return "", fmt.Errorf("park failed: %s", response.Headers["Message"])
	}
	
	return response.Headers["Exten"], nil
}

// GetCoreShowChannels gets list of active channels
func (c *Client) GetCoreShowChannels() ([]map[string]string, error) {
	action := Action{
		Name: "CoreShowChannels",
	}
	
	response, err := c.SendAction(action)
	if err != nil {
		return nil, err
	}
	
	if response.Headers["Response"] != "Success" {
		return nil, fmt.Errorf("failed to get channels: %s", response.Headers["Message"])
	}
	
	var channels []map[string]string
	
	// Read channel events until we get the complete event
	for {
		event, err := c.readResponse()
		if err != nil {
			return nil, err
		}
		
		if event.Name == "CoreShowChannelsComplete" {
			break
		}
		
		if event.Name == "CoreShowChannel" {
			channels = append(channels, event.Headers)
		}
	}
	
	return channels, nil
}

// Close closes the AMI connection
func (c *Client) Close() error {
	c.closed = true
	close(c.events)
	
	// Send logoff
	if c.conn != nil {
		action := Action{Name: "Logoff"}
		c.SendAction(action)
		return c.conn.Close()
	}
	
	return nil
}