package services

import (
	"fmt"
	"strings"
	
	"io/ioutil"
	"os"
	
	"airwavepbx/internal/ami"
	"airwavepbx/internal/database"
)

// Trunk represents a SIP trunk
type Trunk struct {
	ID         int    `json:"id"`
	Name       string `json:"name"`
	Provider   string `json:"provider"`
	Host       string `json:"host"`
	Username   string `json:"username"`
	Password   string `json:"password,omitempty"`
	FromUser   string `json:"from_user"`
	FromDomain string `json:"from_domain"`
	Register   bool   `json:"register"`
	CodecPrefs string `json:"codec_prefs"`
	Enabled    bool   `json:"enabled"`
}

// TrunkService handles trunk-related operations
type TrunkService struct {
	db  *database.DB
	ami *ami.Client
}

// NewTrunkService creates a new trunk service
func NewTrunkService(db *database.DB, ami *ami.Client) *TrunkService {
	return &TrunkService{
		db:  db,
		ami: ami,
	}
}

// GetAll returns all trunks
func (s *TrunkService) GetAll() ([]Trunk, error) {
	query := `
		SELECT id, name, provider, host, username, from_user, from_domain,
		       register, codec_prefs, enabled
		FROM trunks
		ORDER BY name
	`
	
	rows, err := s.db.Query(query)
	if err != nil {
		return nil, err
	}
	defer rows.Close()
	
	var trunks []Trunk
	for rows.Next() {
		var trunk Trunk
		err := rows.Scan(
			&trunk.ID, &trunk.Name, &trunk.Provider, &trunk.Host,
			&trunk.Username, &trunk.FromUser, &trunk.FromDomain,
			&trunk.Register, &trunk.CodecPrefs, &trunk.Enabled,
		)
		if err != nil {
			return nil, err
		}
		trunks = append(trunks, trunk)
	}
	
	return trunks, nil
}

// Get returns a specific trunk
func (s *TrunkService) Get(id int) (*Trunk, error) {
	var trunk Trunk
	query := `
		SELECT id, name, provider, host, username, password, from_user, 
		       from_domain, register, codec_prefs, enabled
		FROM trunks
		WHERE id = ?
	`
	
	err := s.db.QueryRow(query, id).Scan(
		&trunk.ID, &trunk.Name, &trunk.Provider, &trunk.Host,
		&trunk.Username, &trunk.Password, &trunk.FromUser,
		&trunk.FromDomain, &trunk.Register, &trunk.CodecPrefs, &trunk.Enabled,
	)
	
	if err != nil {
		return nil, err
	}
	
	return &trunk, nil
}

// Create adds a new trunk
func (s *TrunkService) Create(trunk *Trunk) error {
	// Set defaults
	if trunk.CodecPrefs == "" {
		trunk.CodecPrefs = "OPUS,G722,PCMU,PCMA"
	}
	
	// Auto-configure based on provider
	s.autoConfigureProvider(trunk)
	
	// Insert into database
	result, err := s.db.Exec(`
		INSERT INTO trunks (name, provider, host, username, password,
		                   from_user, from_domain, register, codec_prefs, enabled)
		VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
	`, trunk.Name, trunk.Provider, trunk.Host, trunk.Username, trunk.Password,
		trunk.FromUser, trunk.FromDomain, trunk.Register, trunk.CodecPrefs, trunk.Enabled)
	
	if err != nil {
		return err
	}
	
	id, _ := result.LastInsertId()
	trunk.ID = int(id)
	
	// Update Asterisk configuration
	return s.updateAsteriskConfig()
}

// Update modifies an existing trunk
func (s *TrunkService) Update(trunk *Trunk) error {
	_, err := s.db.Exec(`
		UPDATE trunks 
		SET name = ?, provider = ?, host = ?, username = ?, password = ?,
		    from_user = ?, from_domain = ?, register = ?, codec_prefs = ?, enabled = ?
		WHERE id = ?
	`, trunk.Name, trunk.Provider, trunk.Host, trunk.Username, trunk.Password,
		trunk.FromUser, trunk.FromDomain, trunk.Register, trunk.CodecPrefs, 
		trunk.Enabled, trunk.ID)
	
	if err != nil {
		return err
	}
	
	// Update Asterisk configuration
	return s.updateAsteriskConfig()
}

// Delete removes a trunk
func (s *TrunkService) Delete(id int) error {
	_, err := s.db.Exec("DELETE FROM trunks WHERE id = ?", id)
	if err != nil {
		return err
	}
	
	// Update Asterisk configuration
	return s.updateAsteriskConfig()
}

// autoConfigureProvider sets provider-specific defaults
func (s *TrunkService) autoConfigureProvider(trunk *Trunk) {
	switch strings.ToLower(trunk.Provider) {
	case "callcentric":
		if trunk.FromDomain == "" {
			trunk.FromDomain = "callcentric.com"
		}
		if trunk.Host == "" {
			trunk.Host = "callcentric.com"
		}
		trunk.Register = true
		
	case "voip.ms", "voipms":
		if trunk.FromDomain == "" {
			trunk.FromDomain = "voip.ms"
		}
		if trunk.Host == "" && trunk.Username != "" {
			// VoIP.ms uses subaccounts with specific servers
			trunk.Host = "atlanta.voip.ms" // Default to Atlanta
		}
		trunk.Register = true
		
	case "twilio":
		trunk.Register = false // Twilio doesn't use registration
		if trunk.FromDomain == "" {
			trunk.FromDomain = "twilio.com"
		}
		
	case "flowroute":
		if trunk.FromDomain == "" {
			trunk.FromDomain = "flowroute.com"
		}
		if trunk.Host == "" {
			trunk.Host = "sip.flowroute.com"
		}
		trunk.Register = false // Flowroute uses IP auth
	}
}

// updateAsteriskConfig generates and applies Asterisk PJSIP trunk configurations
func (s *TrunkService) updateAsteriskConfig() error {
	trunks, err := s.GetAll()
	if err != nil {
		return err
	}
	
	// Build PJSIP trunk configuration
	config := "; AirwavePBX Generated PJSIP Trunks\n\n"
	
	for _, trunk := range trunks {
		if !trunk.Enabled {
			continue
		}
		
		// Create trunk endpoint configuration
		config += fmt.Sprintf(`; Trunk: %s (%s)
[%s]
type=endpoint
context=from-trunk
from_user=%s
from_domain=%s
disallow=all
`,
			trunk.Name, trunk.Provider,
			trunk.Name, trunk.FromUser, trunk.FromDomain)
		
		// Add codecs
		codecs := strings.Split(trunk.CodecPrefs, ",")
		for _, codec := range codecs {
			codec = strings.TrimSpace(strings.ToLower(codec))
			if codec == "opus" || codec == "g722" || codec == "ulaw" || codec == "alaw" {
				config += fmt.Sprintf("allow=%s\n", codec)
			}
		}
		
		// Add registration and authentication
		config += fmt.Sprintf(`auth=%s
aors=%s
outbound_auth=%s
`,
			trunk.Name, trunk.Name, trunk.Name)
		
		// Add media options
		config += `direct_media=no
trust_id_inbound=yes
send_pai=yes
rtp_symmetric=yes
rewrite_contact=yes

`
		
		// Create authentication
		config += fmt.Sprintf(`[%s]
type=auth
auth_type=userpass
username=%s
password=%s

`,
			trunk.Name, trunk.Username, trunk.Password)
		
		// Create AOR (Address of Record)
		config += fmt.Sprintf(`[%s]
type=aor
contact=sip:%s@%s
`,
			trunk.Name, trunk.Username, trunk.Host)
		
		if trunk.Register {
			config += "qualify_frequency=60\n"
		}
		config += "\n"
		
		// Create outbound registration if needed
		if trunk.Register {
			config += fmt.Sprintf(`[%s]
type=registration
transport=transport-udp
outbound_auth=%s
server_uri=sip:%s
client_uri=sip:%s@%s
retry_interval=60
expiration=600

`,
				trunk.Name+"-reg", trunk.Name, trunk.Host, trunk.Username, trunk.Host)
		}
		
		// Create identify section for inbound matching
		config += fmt.Sprintf(`[%s]
type=identify
endpoint=%s
match=%s

`,
			trunk.Name+"-identify", trunk.Name, trunk.Host)
	}
	
	// Write to Asterisk PJSIP configuration
	configPath := "/etc/asterisk/pjsip_airwave_trunks.conf"
	err = ioutil.WriteFile(configPath, []byte(config), 0644)
	if err != nil {
		return fmt.Errorf("failed to write PJSIP trunk config: %w", err)
	}
	
	// Ensure the file is included in the main pjsip.conf
	mainConfig := "/etc/asterisk/pjsip.conf"
	includeDirective := "#include pjsip_airwave_trunks.conf\n"
	
	// Check if include already exists
	content, err := ioutil.ReadFile(mainConfig)
	if err == nil && !strings.Contains(string(content), includeDirective) {
		// Append include directive
		f, err := os.OpenFile(mainConfig, os.O_APPEND|os.O_WRONLY, 0644)
		if err != nil {
			return fmt.Errorf("failed to update main pjsip.conf: %w", err)
		}
		defer f.Close()
		
		if _, err := f.WriteString("\n" + includeDirective); err != nil {
			return fmt.Errorf("failed to write include directive: %w", err)
		}
	}
	
	// Reload PJSIP module in Asterisk
	action := ami.Action{
		Name: "Command",
		Headers: map[string]string{
			"Command": "pjsip reload",
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return fmt.Errorf("failed to reload PJSIP: %w", err)
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to reload PJSIP config: %s", response.Headers["Message"])
	}
	
	return nil
}

// TestTrunk verifies a trunk configuration
func (s *TrunkService) TestTrunk(id int) error {
	trunk, err := s.Get(id)
	if err != nil {
		return err
	}
	
	// Check PJSIP trunk status
	action := ami.Action{
		Name: "Command",
		Headers: map[string]string{
			"Command": fmt.Sprintf("pjsip show endpoint %s", trunk.Name),
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to get trunk status: %s", response.Headers["Message"])
	}
	
	// For registered trunks, check registration status
	if trunk.Register {
		action = ami.Action{
			Name: "Command",
			Headers: map[string]string{
				"Command": fmt.Sprintf("pjsip show registration %s-reg", trunk.Name),
			},
		}
		
		response, err = s.ami.SendAction(action)
		if err != nil {
			return err
		}
		
		if response.Headers["Response"] != "Success" {
			return fmt.Errorf("trunk not registered: %s", response.Headers["Message"])
		}
	}
	
	return nil
}