package services

import (
	"database/sql"
	"fmt"
	"time"
	
	"airwavepbx/internal/ami"
	"airwavepbx/internal/database"
	"github.com/google/uuid"
)

// Call represents a phone call
type Call struct {
	ID               int       `json:"id"`
	UUID             string    `json:"uuid"`
	Direction        string    `json:"direction"`
	CallerIDName     string    `json:"caller_id_name"`
	CallerIDNumber   string    `json:"caller_id_number"`
	DestinationNumber string   `json:"destination_number"`
	StartTime        time.Time `json:"start_time"`
	AnswerTime       *time.Time `json:"answer_time,omitempty"`
	EndTime          *time.Time `json:"end_time,omitempty"`
	Duration         int       `json:"duration"`
	BillSec          int       `json:"billsec"`
	Disposition      string    `json:"disposition"`
	RecordingPath    string    `json:"recording_path,omitempty"`
	TrunkID          *int      `json:"trunk_id,omitempty"`
	ExtensionID      *int      `json:"extension_id,omitempty"`
}

// ActiveCall represents a currently active call
type ActiveCall struct {
	UUID             string    `json:"uuid"`
	Channel          string    `json:"channel"`  // Asterisk channel name
	CallerIDName     string    `json:"caller_id_name"`
	CallerIDNumber   string    `json:"caller_id_number"`
	DestinationNumber string   `json:"destination_number"`
	LineNumber       int       `json:"line_number"`
	State            string    `json:"state"`
	AnsweredBy       string    `json:"answered_by,omitempty"`
	StartTime        time.Time `json:"start_time"`
	AnswerTime       *time.Time `json:"answer_time,omitempty"`
	ConferenceID     string    `json:"conference_id,omitempty"`
	IsMuted          bool      `json:"is_muted"`
	IsOnAir          bool      `json:"is_on_air"`
}

// CallService handles call-related operations
type CallService struct {
	db  *database.DB
	ami *ami.Client
}

// NewCallService creates a new call service
func NewCallService(db *database.DB, ami *ami.Client) *CallService {
	return &CallService{
		db:  db,
		ami: ami,
	}
}

// GetActiveCalls returns all active calls
func (s *CallService) GetActiveCalls() ([]ActiveCall, error) {
	query := `
		SELECT uuid, caller_id_name, caller_id_number, destination_number,
		       line_number, state, answered_by, start_time, answer_time,
		       conference_id, is_muted, is_on_air
		FROM active_calls
		ORDER BY start_time DESC
	`
	
	rows, err := s.db.Query(query)
	if err != nil {
		return nil, err
	}
	defer rows.Close()
	
	var calls []ActiveCall
	for rows.Next() {
		var call ActiveCall
		err := rows.Scan(
			&call.UUID, &call.CallerIDName, &call.CallerIDNumber,
			&call.DestinationNumber, &call.LineNumber, &call.State,
			&call.AnsweredBy, &call.StartTime, &call.AnswerTime,
			&call.ConferenceID, &call.IsMuted, &call.IsOnAir,
		)
		if err != nil {
			return nil, err
		}
		calls = append(calls, call)
	}
	
	return calls, nil
}

// GetCallHistory returns call logs with pagination
func (s *CallService) GetCallHistory(limit, offset int) ([]Call, int, error) {
	// Get total count
	var total int
	err := s.db.QueryRow("SELECT COUNT(*) FROM call_logs").Scan(&total)
	if err != nil {
		return nil, 0, err
	}
	
	// Get calls
	query := `
		SELECT id, uuid, direction, caller_id_name, caller_id_number,
		       destination_number, start_time, answer_time, end_time,
		       duration, billsec, disposition, recording_path,
		       trunk_id, extension_id
		FROM call_logs
		ORDER BY start_time DESC
		LIMIT ? OFFSET ?
	`
	
	rows, err := s.db.Query(query, limit, offset)
	if err != nil {
		return nil, 0, err
	}
	defer rows.Close()
	
	var calls []Call
	for rows.Next() {
		var call Call
		err := rows.Scan(
			&call.ID, &call.UUID, &call.Direction, &call.CallerIDName,
			&call.CallerIDNumber, &call.DestinationNumber, &call.StartTime,
			&call.AnswerTime, &call.EndTime, &call.Duration, &call.BillSec,
			&call.Disposition, &call.RecordingPath, &call.TrunkID,
			&call.ExtensionID,
		)
		if err != nil {
			return nil, 0, err
		}
		calls = append(calls, call)
	}
	
	return calls, total, nil
}

// AnswerCall answers an incoming call
func (s *CallService) AnswerCall(uuid string) error {
	// Update database
	_, err := s.db.Exec(`
		UPDATE active_calls 
		SET state = 'answered', answer_time = CURRENT_TIMESTAMP 
		WHERE uuid = ?
	`, uuid)
	if err != nil {
		return err
	}
	
	// Get channel from database
	var channel string
	err = s.db.QueryRow("SELECT channel FROM active_calls WHERE uuid = ?", uuid).Scan(&channel)
	if err != nil {
		return fmt.Errorf("failed to get channel: %w", err)
	}
	
	// Answer in Asterisk
	action := ami.Action{
		Name: "Answer",
		Headers: map[string]string{
			"Channel": channel,
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to answer call: %s", response.Headers["Message"])
	}
	
	return nil
}

// HangupCall ends a call
func (s *CallService) HangupCall(uuid string) error {
	// Update database
	_, err := s.db.Exec("DELETE FROM active_calls WHERE uuid = ?", uuid)
	if err != nil {
		return err
	}
	
	// Get channel from database
	var channel string
	err = s.db.QueryRow("SELECT channel FROM active_calls WHERE uuid = ?", uuid).Scan(&channel)
	if err != nil {
		return fmt.Errorf("failed to get channel: %w", err)
	}
	
	// Hangup in Asterisk
	return s.ami.HangupCall(channel)
}

// TransferCall transfers a call to another extension
func (s *CallService) TransferCall(uuid, destination string) error {
	// Update database
	_, err := s.db.Exec(`
		UPDATE active_calls 
		SET destination_number = ?, state = 'transferring' 
		WHERE uuid = ?
	`, destination, uuid)
	if err != nil {
		return err
	}
	
	// Get channel from database
	var channel string
	err = s.db.QueryRow("SELECT channel FROM active_calls WHERE uuid = ?", uuid).Scan(&channel)
	if err != nil {
		return fmt.Errorf("failed to get channel: %w", err)
	}
	
	// Transfer in Asterisk
	return s.ami.TransferCall(channel, destination)
}

// HoldCall puts a call on hold
func (s *CallService) HoldCall(uuid string) error {
	// Update database state
	_, err := s.db.Exec(`
		UPDATE active_calls SET state = 'held' WHERE uuid = ?
	`, uuid)
	if err != nil {
		return err
	}
	
	// Get channel from database
	var channel string
	err = s.db.QueryRow("SELECT channel FROM active_calls WHERE uuid = ?", uuid).Scan(&channel)
	if err != nil {
		return fmt.Errorf("failed to get channel: %w", err)
	}
	
	// Hold in Asterisk using MusicOnHold
	action := ami.Action{
		Name: "SetVar",
		Headers: map[string]string{
			"Channel":  channel,
			"Variable": "MUSICCLASS",
			"Value":    "default",
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to hold call: %s", response.Headers["Message"])
	}
	
	// Start music on hold
	action = ami.Action{
		Name: "MusicOnHold",
		Headers: map[string]string{
			"Channel": channel,
			"Class":   "default",
		},
	}
	
	response, err = s.ami.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to start music on hold: %s", response.Headers["Message"])
	}
	
	return nil
}

// UnholdCall resumes a held call
func (s *CallService) UnholdCall(uuid string) error {
	// Update database state
	_, err := s.db.Exec(`
		UPDATE active_calls SET state = 'active' WHERE uuid = ?
	`, uuid)
	if err != nil {
		return err
	}
	
	// Get channel from database
	var channel string
	err = s.db.QueryRow("SELECT channel FROM active_calls WHERE uuid = ?", uuid).Scan(&channel)
	if err != nil {
		return fmt.Errorf("failed to get channel: %w", err)
	}
	
	// Stop music on hold in Asterisk
	action := ami.Action{
		Name: "StopMusicOnHold",
		Headers: map[string]string{
			"Channel": channel,
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to unhold call: %s", response.Headers["Message"])
	}
	
	return nil
}

// ConferenceCall adds a call to a conference
func (s *CallService) ConferenceCall(uuid, conferenceID string) error {
	// Update database
	_, err := s.db.Exec(`
		UPDATE active_calls 
		SET conference_id = ?, state = 'conferenced' 
		WHERE uuid = ?
	`, conferenceID, uuid)
	if err != nil {
		return err
	}
	
	// Get channel from database
	var channel string
	err = s.db.QueryRow("SELECT channel FROM active_calls WHERE uuid = ?", uuid).Scan(&channel)
	if err != nil {
		return fmt.Errorf("failed to get channel: %w", err)
	}
	
	// Transfer to conference in Asterisk
	return s.ami.TransferCall(channel, conferenceID)
}

// MuteCall mutes a call in a conference
func (s *CallService) MuteCall(uuid string) error {
	// Update database
	_, err := s.db.Exec(`
		UPDATE active_calls SET is_muted = 1 WHERE uuid = ?
	`, uuid)
	if err != nil {
		return err
	}
	
	// Get channel from database
	var channel string
	err = s.db.QueryRow("SELECT channel FROM active_calls WHERE uuid = ?", uuid).Scan(&channel)
	if err != nil {
		return fmt.Errorf("failed to get channel: %w", err)
	}
	
	// Mute in Asterisk
	action := ami.Action{
		Name: "MuteAudio",
		Headers: map[string]string{
			"Channel":   channel,
			"Direction": "in",
			"State":     "on",
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to mute call: %s", response.Headers["Message"])
	}
	
	return nil
}

// UnmuteCall unmutes a call in a conference
func (s *CallService) UnmuteCall(uuid string) error {
	// Update database
	_, err := s.db.Exec(`
		UPDATE active_calls SET is_muted = 0 WHERE uuid = ?
	`, uuid)
	if err != nil {
		return err
	}
	
	// Get channel from database
	var channel string
	err = s.db.QueryRow("SELECT channel FROM active_calls WHERE uuid = ?", uuid).Scan(&channel)
	if err != nil {
		return fmt.Errorf("failed to get channel: %w", err)
	}
	
	// Unmute in Asterisk
	action := ami.Action{
		Name: "MuteAudio",
		Headers: map[string]string{
			"Channel":   channel,
			"Direction": "in",
			"State":     "off",
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return err
	}
	
	if response.Headers["Response"] != "Success" {
		return fmt.Errorf("failed to unmute call: %s", response.Headers["Message"])
	}
	
	return nil
}

// SetOnAir marks a call as being on-air
func (s *CallService) SetOnAir(uuid string, onAir bool) error {
	_, err := s.db.Exec(`
		UPDATE active_calls SET is_on_air = ? WHERE uuid = ?
	`, onAir, uuid)
	return err
}

// CreateOutboundCall originates a new call
func (s *CallService) CreateOutboundCall(from, to string) (string, error) {
	// Generate UUID
	callUUID := uuid.New().String()
	
	// Create ActionID for tracking
	actionID := fmt.Sprintf("AIRWAVE_%s", callUUID)
	
	// Originate call in Asterisk
	action := ami.Action{
		Name: "Originate",
		Headers: map[string]string{
			"Channel":   fmt.Sprintf("PJSIP/%s", to),
			"Context":   "internal",
			"Exten":     to,
			"Priority":  "1",
			"CallerID":  fmt.Sprintf("%s <%s>", from, from),
			"Timeout":   "30000",
			"ActionID":  actionID,
			"Variable":  fmt.Sprintf("AIRWAVE_UUID=%s", callUUID),
		},
	}
	
	response, err := s.ami.SendAction(action)
	if err != nil {
		return "", err
	}
	
	if response.Headers["Response"] != "Success" {
		return "", fmt.Errorf("originate failed: %s", response.Headers["Message"])
	}
	
	// Note: We'll insert into active_calls when we receive the Newchannel event
	// This is because we need the channel name from Asterisk
	
	return callUUID, nil
}

// GetCallerInfo retrieves caller information
func (s *CallService) GetCallerInfo(phoneNumber string) (*Caller, error) {
	var caller Caller
	query := `
		SELECT id, phone_number, first_name, last_name, city, state, country,
		       last_called, call_count, notes, tags, banned, vip
		FROM callers
		WHERE phone_number = ?
	`
	
	err := s.db.QueryRow(query, phoneNumber).Scan(
		&caller.ID, &caller.PhoneNumber, &caller.FirstName, &caller.LastName,
		&caller.City, &caller.State, &caller.Country, &caller.LastCalled,
		&caller.CallCount, &caller.Notes, &caller.Tags, &caller.Banned, &caller.VIP,
	)
	
	if err == sql.ErrNoRows {
		// Create new caller record
		caller = Caller{
			PhoneNumber: phoneNumber,
			CallCount:   0,
		}
		
		result, err := s.db.Exec(`
			INSERT INTO callers (phone_number, call_count) VALUES (?, 0)
		`, phoneNumber)
		if err != nil {
			return nil, err
		}
		
		id, _ := result.LastInsertId()
		caller.ID = int(id)
		
		return &caller, nil
	}
	
	if err != nil {
		return nil, err
	}
	
	// Update call count and last called
	_, err = s.db.Exec(`
		UPDATE callers 
		SET call_count = call_count + 1, last_called = CURRENT_TIMESTAMP 
		WHERE id = ?
	`, caller.ID)
	
	return &caller, err
}

// Caller represents a caller's information
type Caller struct {
	ID          int        `json:"id"`
	PhoneNumber string     `json:"phone_number"`
	FirstName   string     `json:"first_name"`
	LastName    string     `json:"last_name"`
	City        string     `json:"city"`
	State       string     `json:"state"`
	Country     string     `json:"country"`
	LastCalled  *time.Time `json:"last_called"`
	CallCount   int        `json:"call_count"`
	Notes       string     `json:"notes"`
	Tags        string     `json:"tags"`
	Banned      bool       `json:"banned"`
	VIP         bool       `json:"vip"`
}