// Settings Page

async function loadSettingsPage() {
    const contentDiv = document.getElementById('page-content');
    
    contentDiv.innerHTML = `
        <div class="page-header">
            <h1 class="page-title">Settings</h1>
            <div class="page-actions">
                <button class="btn btn-primary" onclick="saveSettings()">
                    <i class="fap fa-save"></i>
                    <span>Save Changes</span>
                </button>
            </div>
        </div>
        
        <div class="settings-grid">
            <div class="settings-section glass-card">
                <h2>General Settings</h2>
                <form id="general-settings-form">
                    <div class="form-group">
                        <label>Company Name</label>
                        <div class="glass-input">
                            <i class="fap fa-building input-icon"></i>
                            <input type="text" name="company_name" id="company_name" placeholder="Your Company">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Time Zone</label>
                        <select name="timezone" id="timezone" class="glass-input">
                            <option value="UTC">UTC</option>
                            <option value="America/New_York">Eastern Time</option>
                            <option value="America/Chicago">Central Time</option>
                            <option value="America/Denver">Mountain Time</option>
                            <option value="America/Los_Angeles">Pacific Time</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Max Call Duration (seconds)</label>
                        <div class="glass-input">
                            <i class="fap fa-clock input-icon"></i>
                            <input type="number" name="max_call_duration" id="max_call_duration" 
                                   placeholder="3600" min="60" max="86400">
                        </div>
                    </div>
                </form>
            </div>
            
            <div class="settings-section glass-card">
                <h2>Recording Settings</h2>
                <form id="recording-settings-form">
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input type="checkbox" name="recording_enabled" id="recording_enabled">
                            <span>Enable Call Recording</span>
                        </label>
                    </div>
                    
                    <div class="form-group">
                        <label>Recording Format</label>
                        <select name="recording_format" id="recording_format" class="glass-input">
                            <option value="wav">WAV (Best Quality)</option>
                            <option value="mp3">MP3 (Compressed)</option>
                            <option value="opus">OPUS (Modern)</option>
                        </select>
                    </div>
                </form>
            </div>
            
            <div class="settings-section glass-card">
                <h2>SSL Certificate</h2>
                <div id="ssl-status">
                    <div class="loading">Loading SSL status...</div>
                </div>
                <button class="btn btn-primary mt-3" onclick="showSSLConfig()">
                    <i class="fap fa-lock"></i>
                    <span>Configure SSL</span>
                </button>
            </div>
            
            <div class="settings-section glass-card">
                <h2>System Information</h2>
                <div id="system-info">
                    <div class="loading">Loading system info...</div>
                </div>
            </div>
        </div>
    `;
    
    // Load settings
    await loadSettings();
}

async function loadSettings() {
    try {
        const [settings, status] = await Promise.all([
            API.getSettings(),
            API.getSystemStatus()
        ]);
        
        // Populate form fields
        for (const [key, value] of Object.entries(settings)) {
            const field = document.getElementById(key);
            if (field) {
                if (field.type === 'checkbox') {
                    field.checked = value === true || value === 'true';
                } else {
                    field.value = value;
                }
            }
        }
        
        // Show system info
        document.getElementById('system-info').innerHTML = `
            <div class="info-grid">
                <div class="info-item">
                    <span class="info-label">Version</span>
                    <span class="info-value">${status.version}</span>
                </div>
                <div class="info-item">
                    <span class="info-label">Uptime</span>
                    <span class="info-value">${formatUptime(status.uptime)}</span>
                </div>
                <div class="info-item">
                    <span class="info-label">FreeSWITCH</span>
                    <span class="info-value">
                        <span class="status-indicator ${status.services.freeswitch === 'running' ? 'success' : 'danger'}"></span>
                        ${status.services.freeswitch}
                    </span>
                </div>
                <div class="info-item">
                    <span class="info-label">Database</span>
                    <span class="info-value">
                        <span class="status-indicator ${status.services.database === 'connected' ? 'success' : 'danger'}"></span>
                        ${status.services.database}
                    </span>
                </div>
            </div>
        `;
        
        // Check SSL status
        checkSSLStatus();
        
    } catch (error) {
        showNotification('Error', 'Failed to load settings', 'error');
    }
}

async function saveSettings() {
    const generalForm = document.getElementById('general-settings-form');
    const recordingForm = document.getElementById('recording-settings-form');
    
    const generalData = new FormData(generalForm);
    const recordingData = new FormData(recordingForm);
    
    const settings = {};
    
    // Collect all form data
    for (const [key, value] of generalData.entries()) {
        settings[key] = value;
    }
    
    for (const [key, value] of recordingData.entries()) {
        settings[key] = value;
    }
    
    // Handle checkboxes
    settings.recording_enabled = document.getElementById('recording_enabled').checked;
    
    try {
        await API.updateSettings(settings);
        showNotification('Success', 'Settings saved successfully', 'success');
    } catch (error) {
        showNotification('Error', 'Failed to save settings', 'error');
    }
}

async function checkSSLStatus() {
    // In production, this would check actual SSL certificate status
    const sslStatus = document.getElementById('ssl-status');
    
    // Simulate SSL check
    const hasSSL = window.location.protocol === 'https:';
    
    if (hasSSL) {
        sslStatus.innerHTML = `
            <div class="ssl-info success">
                <i class="fap fa-check-circle"></i>
                <div>
                    <strong>SSL Certificate Active</strong>
                    <p class="text-secondary">Your connection is secure</p>
                </div>
            </div>
        `;
    } else {
        sslStatus.innerHTML = `
            <div class="ssl-info warning">
                <i class="fap fa-exclamation-triangle"></i>
                <div>
                    <strong>No SSL Certificate</strong>
                    <p class="text-secondary">Configure SSL for secure connections</p>
                </div>
            </div>
        `;
    }
}

function showSSLConfig() {
    showModal('Configure SSL Certificate', `
        <form id="ssl-form" onsubmit="handleSSLConfig(event)">
            <div class="form-group">
                <label>Domain Name *</label>
                <div class="glass-input">
                    <i class="fap fa-globe input-icon"></i>
                    <input type="text" name="domain" placeholder="pbx.yourdomain.com" required>
                </div>
                <small class="text-secondary">
                    Make sure this domain points to your server's IP address
                </small>
            </div>
            
            <div class="form-group">
                <label>Email Address *</label>
                <div class="glass-input">
                    <i class="fap fa-envelope input-icon"></i>
                    <input type="email" name="email" placeholder="admin@yourdomain.com" required>
                </div>
                <small class="text-secondary">
                    Used for Let's Encrypt notifications
                </small>
            </div>
            
            <div class="alert alert-info mt-3">
                <i class="fap fa-info-circle"></i>
                This will use Let's Encrypt to generate a free SSL certificate.
                Make sure port 80 is accessible from the internet.
            </div>
        </form>
    `, [
        {
            label: 'Cancel',
            onclick: 'closeModal()'
        },
        {
            label: 'Generate Certificate',
            class: 'btn-primary',
            icon: 'lock',
            onclick: "document.getElementById('ssl-form').submit()"
        }
    ]);
}

async function handleSSLConfig(event) {
    event.preventDefault();
    const formData = new FormData(event.target);
    
    // In production, this would trigger SSL certificate generation
    showNotification('Info', 'SSL configuration would be processed here', 'info');
    closeModal();
}

// Add settings-specific styles
const settingsStyles = `
<style>
.settings-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: var(--spacing-lg);
}

.settings-section {
    padding: var(--spacing-lg);
}

.settings-section h2 {
    font-size: 20px;
    margin-bottom: var(--spacing-lg);
    padding-bottom: var(--spacing-md);
    border-bottom: 1px solid var(--glass-border);
}

.info-grid {
    display: grid;
    gap: var(--spacing-md);
}

.info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.info-label {
    color: var(--text-secondary);
    font-size: 14px;
}

.info-value {
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: var(--spacing-xs);
}

.ssl-info {
    display: flex;
    align-items: start;
    gap: var(--spacing-md);
    padding: var(--spacing-md);
    background: var(--glass-bg);
    border-radius: var(--radius-md);
}

.ssl-info i {
    font-size: 24px;
    margin-top: 4px;
}

.ssl-info.success i {
    color: var(--success-color);
}

.ssl-info.warning i {
    color: var(--warning-color);
}

.ssl-info strong {
    display: block;
    margin-bottom: 4px;
}

.ssl-info p {
    margin: 0;
    font-size: 13px;
}

.alert {
    padding: var(--spacing-md);
    background: var(--glass-bg);
    border: 1px solid var(--glass-border);
    border-radius: var(--radius-md);
    display: flex;
    align-items: start;
    gap: var(--spacing-sm);
}

.alert-info {
    border-color: rgba(0, 122, 255, 0.3);
    background: rgba(0, 122, 255, 0.1);
}

.alert i {
    color: var(--primary-color);
    flex-shrink: 0;
}

@media (max-width: 768px) {
    .settings-grid {
        grid-template-columns: 1fr;
    }
}
</style>
`;

if (!document.getElementById('settings-styles')) {
    const styleElement = document.createElement('div');
    styleElement.id = 'settings-styles';
    styleElement.innerHTML = settingsStyles;
    document.head.appendChild(styleElement);
}