#!/bin/bash
# AirwavePBX Professional Installation Script
# Version: 1.0.0 - FreeSWITCH-based architecture
# Target OS: Ubuntu 24.04 LTS

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
MAGENTA='\033[0;35m'
NC='\033[0m'
BOLD='\033[1m'

# Script version
INSTALLER_VERSION="1.0.0"
AIRWAVE_VERSION="1.0.0"

# Configuration
INSTALL_DIR="/opt/airwavepbx"
DATA_DIR="/var/lib/airwavepbx"
LOG_DIR="/var/log/airwavepbx"
CREDENTIALS_FILE="/root/airwavepbx-credentials.txt"
SYSTEMD_SERVICE="airwavepbx.service"
SSL_DIR="/etc/letsencrypt"
BACKUP_DIR="/var/backups/airwavepbx"

# Generate secure passwords
DB_PASSWORD=$(openssl rand -hex 32)
ADMIN_PASSWORD=$(openssl rand -base64 12 | tr -d "=+/" | cut -c1-16)
API_KEY=$(openssl rand -hex 32)
ESL_PASSWORD=$(openssl rand -hex 16)

# Print banner
print_banner() {
    clear
    echo -e "${CYAN}"
    cat << "EOF"
⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿
⣿⣿⣿⣿⣿⣿⣿⡏⠀⠀⢻⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿
⣿⣿⣿⣿⣿⣿⡟⠀⣰⡆⡘⠁⠹⣿⣿⣿⣿⣿⣿⣿⣿⠟⠛⣿⣿⡏⠉⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⡟⠛⠛⠛⠻⣿⡟⠛⠛⠛⠻⣟⠛⢻⣿⡟⠛⣿⣿⣿⣿
⣿⣿⣿⣿⣿⡿⠀⢰⣿⣿⣅⠀⢴⣿⣿⣿⣿⣿⣿⣿⡏⢀⡄⠸⣿⡏⠉⣿⠉⢉⣩⠉⢿⡟⠉⢻⡿⠉⡏⣉⣉⠙⢏⠉⢿⡿⠉⡝⠉⣉⡉⠻⡇⠀⣶⣶⠄⢸⡇⠸⠿⠗⠀⣼⣧⡀⠉⢀⣾⣿⣿⣿⣿
⣿⣿⣿⣿⡿⠁⢠⡿⡙⣿⣿⣆⠈⢿⡿⣿⣿⣿⣿⡟⠀⣈⣉⡀⠹⡇⠀⣿⠀⣿⣿⡆⠘⠀⣄⠈⠁⣸⠋⢡⣤⠀⢸⣆⠘⠁⣰⡄⠠⣤⡤⢤⡇⠀⣀⣀⣤⣾⡇⢰⣶⣶⠀⢸⡟⠁⣀⠈⢿⣿⣿⣿⣿
⣿⣿⣿⣿⠁⢠⣭⣣⣿⡸⣱⣶⣆⠀⠀⠘⣿⣿⣿⣤⣼⣿⣿⣷⣤⣥⣤⣿⣤⣿⣿⣿⣤⣼⣿⣦⣴⣿⣦⣤⣤⣤⣼⣿⣤⣴⣿⣷⣦⣤⣤⣾⣧⣤⣿⣿⣿⣿⣧⣤⣤⣤⣴⣮⣤⣾⣿⣧⣤⣽⣿⣿⣿
⣿⣿⣿⠃⢀⣾⣿⣿⣿⣿⣿⣿⣷⡉⠲⣾⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿
⣿⣿⣧⣤⣾⣿⣿⣿⣿⣿⣿⣿⣿⣷⣤⣽⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿
EOF
    echo -e "${NC}"
    echo ""
    echo -e "${BLUE}═════════════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BLUE}                             ${BOLD}AirwavePBX${NC}"
    echo -e "${BLUE}                    ${BOLD}Professional Broadcasting PBX${NC}"
    echo -e "${BLUE}                           Version ${BOLD}${AIRWAVE_VERSION}${NC}"
    echo -e "${BLUE}              ${BOLD}The Ultimate Phone System for Radio Stations${NC}"
    echo -e "${BLUE}═════════════════════════════════════════════════════════════════════════${NC}"
    echo ""
}

# Logging functions
print_step() {
    echo -e "\n${MAGENTA}▶${NC} ${BOLD}$1${NC}"
}

print_info() {
    echo -e "  ${BLUE}ℹ${NC} $1"
}

print_success() {
    echo -e "  ${GREEN}✓${NC} $1"
}

print_error() {
    echo -e "  ${RED}✗${NC} $1"
}

print_warning() {
    echo -e "  ${YELLOW}⚠${NC} $1"
}

# Progress bar
show_progress() {
    local current=$1
    local total=$2
    local width=50
    local percentage=$((current * 100 / total))
    local completed=$((width * current / total))
    
    printf "\r  ["
    printf "%${completed}s" | tr ' ' '='
    printf "%$((width - completed))s" | tr ' ' '-'
    printf "] %3d%%" $percentage
}

# Check if running as root
check_root() {
    if [[ $EUID -ne 0 ]]; then
        print_error "This script must be run as root"
        exit 1
    fi
}

# Detect OS
detect_os() {
    print_step "Detecting Operating System"
    
    if [[ -f /etc/os-release ]]; then
        . /etc/os-release
        OS=$ID
        VER=$VERSION_ID
        print_info "Detected: $PRETTY_NAME"
    else
        print_error "Cannot detect OS"
        exit 1
    fi
    
    if [[ "$OS" == "ubuntu" ]] && [[ "$VER" == "24.04" ]]; then
        print_success "Ubuntu 24.04 LTS detected - Perfect!"
    elif [[ "$OS" == "debian" ]] && [[ "$VER" == "12" ]]; then
        print_warning "Debian 12 detected - Will work but Ubuntu 24.04 LTS recommended"
    else
        print_error "This installer requires Ubuntu 24.04 LTS or Debian 12"
        read -p "Continue anyway? [y/N] " -n 1 -r
        echo
        if [[ ! $REPLY =~ ^[Yy]$ ]]; then
            exit 1
        fi
    fi
}

# Backup existing data
backup_existing_data() {
    if [[ -d "$INSTALL_DIR" ]] || [[ -d "$SSL_DIR" ]]; then
        print_step "Backing Up Existing Data"
        
        # Create backup directory
        mkdir -p "$BACKUP_DIR"
        local backup_name="backup-$(date +%Y%m%d-%H%M%S)"
        local backup_path="$BACKUP_DIR/$backup_name"
        mkdir -p "$backup_path"
        
        # Backup SSL certificates if they exist
        if [[ -d "$SSL_DIR/live" ]]; then
            print_info "Backing up SSL certificates..."
            cp -r "$SSL_DIR" "$backup_path/"
            print_success "SSL certificates backed up"
        fi
        
        # Backup data directory
        if [[ -d "$DATA_DIR" ]]; then
            print_info "Backing up data directory..."
            cp -r "$DATA_DIR" "$backup_path/"
            print_success "Data directory backed up"
        fi
        
        # Backup configuration
        if [[ -f "$INSTALL_DIR/config.yaml" ]]; then
            print_info "Backing up configuration..."
            cp "$INSTALL_DIR/config.yaml" "$backup_path/"
            print_success "Configuration backed up"
        fi
        
        print_success "Backup completed: $backup_path"
    fi
}

# Restore SSL certificates
restore_ssl_certificates() {
    # Find most recent backup with SSL certificates
    local latest_ssl_backup=""
    for backup in $(ls -t "$BACKUP_DIR" 2>/dev/null); do
        if [[ -d "$BACKUP_DIR/$backup/letsencrypt/live" ]]; then
            latest_ssl_backup="$BACKUP_DIR/$backup/letsencrypt"
            break
        fi
    done
    
    if [[ -n "$latest_ssl_backup" ]]; then
        print_info "Restoring SSL certificates from backup..."
        cp -r "$latest_ssl_backup"/* "$SSL_DIR/"
        print_success "SSL certificates restored"
        return 0
    fi
    
    return 1
}

# Install system dependencies
install_dependencies() {
    print_step "Installing System Dependencies"
    
    # Update package lists
    print_info "Updating package lists..."
    apt-get update -qq
    
    # Install essential packages
    local packages=(
        # Build tools
        build-essential
        git
        curl
        wget
        
        # FreeSWITCH dependencies
        autoconf
        automake
        libtool
        pkg-config
        libtool-bin
        
        # Audio libraries
        libspeex-dev
        libspeexdsp-dev
        libopus-dev
        libsndfile-dev
        
        # Database
        sqlite3
        libsqlite3-dev
        
        # SSL/Security
        openssl
        libssl-dev
        ca-certificates
        certbot
        python3-certbot
        
        # System utilities
        systemd
        rsyslog
        logrotate
        htop
        net-tools
        dnsutils
        
        # Firewall
        ufw
        fail2ban
    )
    
    local total=${#packages[@]}
    local current=0
    
    for package in "${packages[@]}"; do
        ((current++))
        show_progress $current $total
        apt-get install -y $package &>/dev/null || print_warning "Failed to install $package"
    done
    
    echo # New line after progress bar
    print_success "Dependencies installed"
}

# Install FreeSWITCH
install_freeswitch() {
    print_step "Installing FreeSWITCH"
    
    # Add FreeSWITCH GPG key
    print_info "Adding FreeSWITCH repository..."
    wget -O - https://files.freeswitch.org/repo/deb/debian-release/fsstretch-archive-keyring.asc | apt-key add - &>/dev/null
    
    # Add FreeSWITCH repository
    echo "deb http://files.freeswitch.org/repo/deb/debian-release/ $(lsb_release -sc) main" > /etc/apt/sources.list.d/freeswitch.list
    echo "deb-src http://files.freeswitch.org/repo/deb/debian-release/ $(lsb_release -sc) main" >> /etc/apt/sources.list.d/freeswitch.list
    
    # Update and install
    apt-get update -qq
    print_info "Installing FreeSWITCH (this may take a few minutes)..."
    apt-get install -y freeswitch-meta-all &>/dev/null
    
    # Stop FreeSWITCH to configure it
    systemctl stop freeswitch &>/dev/null || true
    
    print_success "FreeSWITCH installed"
}

# Configure FreeSWITCH for broadcasting
configure_freeswitch() {
    print_step "Configuring FreeSWITCH for Broadcasting"
    
    # Backup original configs
    print_info "Backing up original configuration..."
    cp -r /etc/freeswitch /etc/freeswitch.backup
    
    # Configure ESL for API access
    print_info "Configuring Event Socket Layer..."
    cat > /etc/freeswitch/autoload_configs/event_socket.conf.xml << EOF
<configuration name="event_socket.conf" description="Socket Client">
  <settings>
    <param name="nat-map" value="false"/>
    <param name="listen-ip" value="127.0.0.1"/>
    <param name="listen-port" value="8021"/>
    <param name="password" value="$ESL_PASSWORD"/>
  </settings>
</configuration>
EOF
    
    # Configure User-Agent string
    print_info "Configuring SIP User-Agent..."
    cat > /etc/freeswitch/autoload_configs/switch.conf.xml << EOF
<configuration name="switch.conf" description="Core Configuration">
  <settings>
    <param name="colorize-console" value="true"/>
    <param name="max-sessions" value="1000"/>
    <param name="sessions-per-second" value="30"/>
    <param name="default-password" value="\$\${default_password}"/>
    <param name="switchname" value="AirwavePBX (FreeSWITCH)"/>
  </settings>
</configuration>
EOF
    
    # Configure SIP profiles for providers
    print_info "Configuring SIP profiles..."
    mkdir -p /etc/freeswitch/sip_profiles/external
    
    # Configure external SIP profile with custom User-Agent
    cat > /etc/freeswitch/sip_profiles/external.xml << EOF
<profile name="external">
  <gateways>
    <X-PRE-PROCESS cmd="include" data="external/*.xml"/>
  </gateways>
  
  <settings>
    <param name="debug" value="0"/>
    <param name="sip-trace" value="no"/>
    <param name="sip-capture" value="no"/>
    <param name="rfc2833-pt" value="101"/>
    <param name="sip-port" value="5080"/>
    <param name="dialplan" value="XML"/>
    <param name="context" value="public"/>
    <param name="dtmf-duration" value="2000"/>
    <param name="inbound-codec-prefs" value="OPUS,G722,PCMU,PCMA"/>
    <param name="outbound-codec-prefs" value="OPUS,G722,PCMU,PCMA"/>
    <param name="rtp-timer-name" value="soft"/>
    <param name="local-network-acl" value="localnet.auto"/>
    <param name="manage-presence" value="false"/>
    <param name="inbound-codec-negotiation" value="generous"/>
    <param name="nonce-ttl" value="60"/>
    <param name="auth-calls" value="false"/>
    <param name="inbound-late-negotiation" value="true"/>
    <param name="inbound-zrtp-auto-secure" value="true"/>
    <param name="rtp-ip" value="\$\${local_ip_v4}"/>
    <param name="sip-ip" value="\$\${local_ip_v4}"/>
    <param name="ext-rtp-ip" value="auto-nat"/>
    <param name="ext-sip-ip" value="auto-nat"/>
    <param name="rtp-timeout-sec" value="300"/>
    <param name="rtp-hold-timeout-sec" value="1800"/>
    <param name="user-agent-string" value="AirwavePBX (FreeSWITCH)"/>
  </settings>
</profile>
EOF
    
    # Create broadcasting-specific dialplan
    print_info "Creating broadcasting dialplan..."
    cat > /etc/freeswitch/dialplan/airwave_broadcasting.xml << EOF
<?xml version="1.0" encoding="utf-8"?>
<include>
  <context name="airwave_broadcasting">
    <!-- Radio Call-In Lines -->
    <extension name="radio_callin">
      <condition field="destination_number" expression="^(1\d{3})$">
        <action application="set" data="call_line=$1"/>
        <action application="set" data="caller_name=${caller_id_name}"/>
        <action application="set" data="caller_number=${caller_id_number}"/>
        
        <!-- Send event to AirwavePBX -->
        <action application="event" data="Event-Name=AIRWAVE::INCOMING_CALL,Line=$1,Caller-ID-Name=${caller_id_name},Caller-ID-Number=${caller_id_number}"/>
        
        <!-- Park the call for screening -->
        <action application="park"/>
      </condition>
    </extension>
  </context>
</include>
EOF
    
    # Set permissions
    chown -R freeswitch:freeswitch /etc/freeswitch
    
    print_success "FreeSWITCH configured for broadcasting"
}

# Install AirwavePBX
install_airwavepbx() {
    print_step "Installing AirwavePBX"
    
    # Create directories
    print_info "Creating directories..."
    mkdir -p $INSTALL_DIR $DATA_DIR $LOG_DIR
    
    # Extract binary and assets
    print_info "Installing AirwavePBX binary..."
    cp airwavepbx $INSTALL_DIR/
    chmod +x $INSTALL_DIR/airwavepbx
    
    # Copy web assets
    print_info "Installing web interface..."
    cp -r web $INSTALL_DIR/
    
    # Copy FAP icons (renamed from fontawesome-pro)
    print_info "Installing icon assets..."
    cp -r fap $INSTALL_DIR/web/static/
    
    # Copy logo assets
    cp -r logos $INSTALL_DIR/web/static/
    
    # Create configuration file
    print_info "Creating configuration..."
    cat > $INSTALL_DIR/config.yaml << EOF
# AirwavePBX Configuration
version: $AIRWAVE_VERSION

server:
  host: 0.0.0.0
  port: 8080
  tls_enabled: true
  tls_cert: /etc/letsencrypt/live/\${DOMAIN}/fullchain.pem
  tls_key: /etc/letsencrypt/live/\${DOMAIN}/privkey.pem

database:
  path: $DATA_DIR/airwavepbx.db
  
freeswitch:
  esl_host: 127.0.0.1
  esl_port: 8021
  esl_password: $ESL_PASSWORD
  
api:
  key: $API_KEY
  cors_origins:
    - https://itsyourradio.com
    - https://broadcaster.itsyourradio.com
    
logging:
  level: info
  path: $LOG_DIR/airwavepbx.log
  max_size: 100
  max_backups: 7
  
admin:
  username: admin
  password: $ADMIN_PASSWORD
EOF
    
    # Set permissions
    chmod 600 $INSTALL_DIR/config.yaml
    
    print_success "AirwavePBX installed"
}

# Create systemd service
create_systemd_service() {
    print_step "Creating System Service"
    
    cat > /etc/systemd/system/$SYSTEMD_SERVICE << EOF
[Unit]
Description=AirwavePBX Professional Broadcasting PBX
After=network.target freeswitch.service
Requires=freeswitch.service

[Service]
Type=simple
User=root
Group=root
WorkingDirectory=$INSTALL_DIR
ExecStart=$INSTALL_DIR/airwavepbx
Restart=always
RestartSec=10
StandardOutput=append:$LOG_DIR/airwavepbx.log
StandardError=append:$LOG_DIR/airwavepbx.log

# Security
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF
    
    systemctl daemon-reload
    systemctl enable $SYSTEMD_SERVICE
    
    print_success "System service created"
}

# Configure firewall
configure_firewall() {
    print_step "Configuring Firewall"
    
    # Enable UFW
    ufw --force enable &>/dev/null
    
    # Allow SSH (preserve existing SSH rules)
    ufw allow ssh &>/dev/null
    
    # Web ports
    ufw allow 80/tcp comment "HTTP" &>/dev/null
    ufw allow 443/tcp comment "HTTPS" &>/dev/null
    ufw allow 8080/tcp comment "AirwavePBX" &>/dev/null
    
    # SIP ports
    ufw allow 5060/udp comment "SIP UDP" &>/dev/null
    ufw allow 5061/tcp comment "SIP TLS" &>/dev/null
    
    # RTP ports for media
    ufw allow 16384:32768/udp comment "RTP Media" &>/dev/null
    
    # Reload firewall
    ufw reload &>/dev/null
    
    print_success "Firewall configured"
}

# Configure log rotation
configure_logging() {
    print_step "Configuring Log Rotation"
    
    cat > /etc/logrotate.d/airwavepbx << EOF
$LOG_DIR/*.log {
    daily
    missingok
    rotate 14
    compress
    notifempty
    create 0640 root root
    sharedscripts
    postrotate
        systemctl reload airwavepbx > /dev/null 2>&1 || true
    endscript
}

/var/log/freeswitch/*.log {
    daily
    missingok
    rotate 7
    compress
    notifempty
    create 0640 freeswitch freeswitch
    sharedscripts
    postrotate
        /usr/bin/fs_cli -x 'fsctl send_sighup' > /dev/null 2>&1 || true
    endscript
}
EOF
    
    print_success "Log rotation configured"
}

# Configure SSL with Let's Encrypt
configure_ssl() {
    print_step "Configuring SSL Certificate"
    
    # Check if we restored certificates
    if restore_ssl_certificates; then
        print_success "Using restored SSL certificates"
        setup_ssl_renewal
        return
    fi
    
    # Ask for domain configuration
    print_info "SSL Certificate Setup (Let's Encrypt)"
    echo ""
    read -p "Enter your domain name (e.g., pbx.yourdomain.com): " DOMAIN
    read -p "Enter your email address for Let's Encrypt notifications: " EMAIL
    
    if [[ -z "$DOMAIN" ]] || [[ -z "$EMAIL" ]]; then
        print_warning "Skipping SSL configuration - domain and email required"
        return
    fi
    
    # Stop AirwavePBX temporarily to free port 80
    systemctl stop airwavepbx 2>/dev/null || true
    
    # Get certificate
    print_info "Obtaining SSL certificate from Let's Encrypt..."
    certbot certonly \
        --standalone \
        --non-interactive \
        --agree-tos \
        --email "$EMAIL" \
        --domains "$DOMAIN" \
        --keep-until-expiring \
        --expand
    
    if [[ $? -eq 0 ]]; then
        print_success "SSL certificate obtained successfully"
        
        # Update configuration with domain
        sed -i "s/\${DOMAIN}/$DOMAIN/g" $INSTALL_DIR/config.yaml
        
        # Copy certificates to AirwavePBX readable location
        mkdir -p $INSTALL_DIR/ssl
        cp "$SSL_DIR/live/$DOMAIN/fullchain.pem" $INSTALL_DIR/ssl/
        cp "$SSL_DIR/live/$DOMAIN/privkey.pem" $INSTALL_DIR/ssl/
        chmod 600 $INSTALL_DIR/ssl/*
        
        setup_ssl_renewal
    else
        print_error "Failed to obtain SSL certificate"
        print_warning "You can configure SSL later by running: airwavepbx configure-ssl"
    fi
}

# Setup SSL auto-renewal
setup_ssl_renewal() {
    print_info "Setting up SSL auto-renewal..."
    
    # Create renewal hook script
    cat > /etc/letsencrypt/renewal-hooks/deploy/airwavepbx.sh << 'EOF'
#!/bin/bash
# Copy renewed certificates to AirwavePBX
DOMAIN=$(basename $(dirname $RENEWED_LINEAGE))
cp "$RENEWED_LINEAGE/fullchain.pem" /opt/airwavepbx/ssl/
cp "$RENEWED_LINEAGE/privkey.pem" /opt/airwavepbx/ssl/
chmod 600 /opt/airwavepbx/ssl/*
systemctl reload airwavepbx
EOF
    
    chmod +x /etc/letsencrypt/renewal-hooks/deploy/airwavepbx.sh
    
    # Test renewal
    certbot renew --dry-run &>/dev/null
    
    if [[ $? -eq 0 ]]; then
        print_success "SSL auto-renewal configured"
    else
        print_warning "SSL auto-renewal test failed - check configuration"
    fi
}

# Save credentials
save_credentials() {
    print_step "Saving Credentials"
    
    cat > $CREDENTIALS_FILE << EOF
════════════════════════════════════════════════════════════════════════
                        AirwavePBX Credentials
                    Generated: $(date '+%Y-%m-%d %H:%M:%S')
════════════════════════════════════════════════════════════════════════

Web Interface:
  URL:      https://YOUR_DOMAIN:8080
  Username: admin
  Password: $ADMIN_PASSWORD

API Access:
  Endpoint: https://YOUR_DOMAIN:8080/api/v1
  API Key:  $API_KEY

FreeSWITCH ESL:
  Host:     127.0.0.1
  Port:     8021
  Password: $ESL_PASSWORD

Database:
  Type:     SQLite
  Path:     $DATA_DIR/airwavepbx.db

════════════════════════════════════════════════════════════════════════

IMPORTANT: Keep this file secure! It contains all your system passwords.
Location: $CREDENTIALS_FILE

EOF
    
    chmod 600 $CREDENTIALS_FILE
    print_success "Credentials saved to $CREDENTIALS_FILE"
}

# Start services
start_services() {
    print_step "Starting Services"
    
    print_info "Starting FreeSWITCH..."
    systemctl start freeswitch
    sleep 5
    
    print_info "Starting AirwavePBX..."
    systemctl start airwavepbx
    sleep 3
    
    # Check if services are running
    if systemctl is-active --quiet freeswitch && systemctl is-active --quiet airwavepbx; then
        print_success "All services started successfully"
    else
        print_error "Services failed to start. Check logs:"
        print_info "  journalctl -u freeswitch -n 50"
        print_info "  journalctl -u airwavepbx -n 50"
    fi
}

# Display completion message
display_completion() {
    echo ""
    echo -e "${GREEN}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
    echo -e "${GREEN}${BOLD}                    🎉 INSTALLATION COMPLETE! 🎉                        ${NC}"
    echo -e "${GREEN}${BOLD}                          AirwavePBX                                    ${NC}"
    echo -e "${GREEN}${BOLD}              Your Broadcasting PBX is Ready for Production              ${NC}"
    echo -e "${GREEN}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}Access Information:${NC}"
    echo -e "  ${BOLD}Web Interface:${NC} https://YOUR_DOMAIN:8080"
    echo -e "  ${BOLD}Username:${NC}      admin"
    echo -e "  ${BOLD}Password:${NC}      ${YELLOW}$ADMIN_PASSWORD${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}API Integration:${NC}"
    echo -e "  ${BOLD}Endpoint:${NC}      https://YOUR_DOMAIN:8080/api/v1"
    echo -e "  ${BOLD}API Key:${NC}       ${YELLOW}$API_KEY${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}Important Files:${NC}"
    echo -e "  ${BOLD}Credentials:${NC}   $CREDENTIALS_FILE"
    echo -e "  ${BOLD}Configuration:${NC} $INSTALL_DIR/config.yaml"
    echo -e "  ${BOLD}Logs:${NC}          $LOG_DIR/"
    echo ""
    if [[ -d "$SSL_DIR/live" ]]; then
        echo -e "${GREEN}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
        echo -e "${GREEN}${BOLD}  ✓ SSL Certificate configured and active with auto-renewal enabled!   ${NC}"
        echo -e "${GREEN}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
    else
        echo -e "${RED}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
        echo -e "${RED}${BOLD}  IMPORTANT: Configure your domain and SSL certificate immediately!     ${NC}"
        echo -e "${RED}${BOLD}  Run: airwavepbx configure-ssl YOUR_DOMAIN.COM                       ${NC}"
        echo -e "${RED}${BOLD}════════════════════════════════════════════════════════════════════════${NC}"
    fi
    echo ""
}

# Main installation flow
main() {
    print_banner
    check_root
    detect_os
    
    print_info "Starting AirwavePBX installation..."
    print_info "This will install FreeSWITCH and all required components"
    echo ""
    read -p "Continue with installation? [Y/n] " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Nn]$ ]]; then
        print_info "Installation cancelled"
        exit 0
    fi
    
    # Backup existing data first
    backup_existing_data
    
    install_dependencies
    install_freeswitch
    configure_freeswitch
    install_airwavepbx
    create_systemd_service
    configure_firewall
    configure_logging
    configure_ssl
    save_credentials
    start_services
    display_completion
}

# Run main function
main "$@"