package api

import (
	"airwavepbx/internal/auth"
	"airwavepbx/internal/services"
	"airwavepbx/internal/websocket"
	"github.com/gofiber/fiber/v2"
)

// SetupRoutes configures all API routes
func SetupRoutes(
	app fiber.Router,
	authService *auth.Service,
	callService *services.CallService,
	extensionService *services.ExtensionService,
	trunkService *services.TrunkService,
	wsHub *websocket.Hub,
) {
	// Public routes
	app.Post("/auth/login", handleLogin(authService))
	
	// Protected routes - require authentication
	api := app.Use(authMiddleware(authService))
	
	// Auth endpoints
	api.Post("/auth/logout", handleLogout())
	api.Get("/auth/me", handleGetCurrentUser())
	
	// Call endpoints
	api.Get("/calls/active", handleGetActiveCalls(callService))
	api.Get("/calls/history", handleGetCallHistory(callService))
	api.Post("/calls/:uuid/answer", handleAnswerCall(callService))
	api.Post("/calls/:uuid/hangup", handleHangupCall(callService))
	api.Post("/calls/:uuid/hold", handleHoldCall(callService))
	api.Post("/calls/:uuid/unhold", handleUnholdCall(callService))
	api.Post("/calls/:uuid/transfer", handleTransferCall(callService))
	api.Post("/calls/:uuid/conference", handleConferenceCall(callService))
	api.Post("/calls/:uuid/mute", handleMuteCall(callService))
	api.Post("/calls/:uuid/unmute", handleUnmuteCall(callService))
	api.Post("/calls/:uuid/on-air", handleSetOnAir(callService))
	api.Post("/calls/dial", handleDialCall(callService))
	api.Get("/calls/:uuid/caller-info", handleGetCallerInfo(callService))
	
	// Extension endpoints
	api.Get("/extensions", handleGetExtensions(extensionService))
	api.Get("/extensions/:id", handleGetExtension(extensionService))
	api.Post("/extensions", handleCreateExtension(extensionService))
	api.Put("/extensions/:id", handleUpdateExtension(extensionService))
	api.Delete("/extensions/:id", handleDeleteExtension(extensionService))
	
	// Trunk endpoints
	api.Get("/trunks", handleGetTrunks(trunkService))
	api.Get("/trunks/:id", handleGetTrunk(trunkService))
	api.Post("/trunks", handleCreateTrunk(trunkService))
	api.Put("/trunks/:id", handleUpdateTrunk(trunkService))
	api.Delete("/trunks/:id", handleDeleteTrunk(trunkService))
	api.Post("/trunks/:id/test", handleTestTrunk(trunkService))
	
	// System endpoints
	api.Get("/system/status", handleGetSystemStatus())
	api.Get("/system/settings", handleGetSettings())
	api.Put("/system/settings", handleUpdateSettings())
	
	// API documentation endpoint
	api.Get("/docs", handleGetAPIDocs())
}