// AirwavePBX Main Application

// Global app state
const App = {
    currentPage: 'dashboard',
    user: null,
    isAuthenticated: false
};

// Initialize application
document.addEventListener('DOMContentLoaded', async () => {
    // Check authentication
    const token = localStorage.getItem('airwave_token');
    if (token) {
        try {
            App.user = await API.getMe();
            App.isAuthenticated = true;
            showMainApp();
        } catch (error) {
            console.error('Auth check failed:', error);
            showLoginScreen();
        }
    } else {
        showLoginScreen();
    }
    
    // Setup event handlers
    setupEventHandlers();
});

// Show login screen
function showLoginScreen() {
    document.getElementById('login-screen').style.display = 'flex';
    document.getElementById('main-app').style.display = 'none';
}

// Show main application
function showMainApp() {
    document.getElementById('login-screen').style.display = 'none';
    document.getElementById('main-app').style.display = 'flex';
    
    // Update user info
    document.getElementById('current-user').textContent = App.user.username;
    
    // Load default page
    loadPage('dashboard');
    
    // Start WebSocket connection
    if (!WS.isConnected) {
        WS.connect();
    }
    
    // Load initial data
    updateActiveCallsCount();
}

// Setup event handlers
function setupEventHandlers() {
    // Login form
    const loginForm = document.getElementById('login-form');
    loginForm.addEventListener('submit', handleLogin);
    
    // Logout button
    const logoutBtn = document.getElementById('logout-btn');
    logoutBtn.addEventListener('click', handleLogout);
    
    // Navigation links
    const navLinks = document.querySelectorAll('.nav-link');
    navLinks.forEach(link => {
        link.addEventListener('click', (e) => {
            e.preventDefault();
            const page = link.dataset.page;
            loadPage(page);
            
            // Update active state
            navLinks.forEach(l => l.classList.remove('active'));
            link.classList.add('active');
        });
    });
}

// Handle login
async function handleLogin(e) {
    e.preventDefault();
    
    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;
    const errorDiv = document.getElementById('login-error');
    
    try {
        errorDiv.textContent = '';
        const response = await API.login(username, password);
        App.user = response.user;
        App.isAuthenticated = true;
        showMainApp();
    } catch (error) {
        errorDiv.textContent = error.message || 'Login failed';
    }
}

// Handle logout
async function handleLogout() {
    try {
        await API.logout();
    } catch (error) {
        console.error('Logout error:', error);
    }
    
    App.user = null;
    App.isAuthenticated = false;
    WS.disconnect();
    showLoginScreen();
}

// Load page
async function loadPage(page) {
    App.currentPage = page;
    const contentDiv = document.getElementById('page-content');
    
    // Show loading
    contentDiv.innerHTML = '<div class="loading">Loading...</div>';
    
    try {
        // Load page module
        switch (page) {
            case 'dashboard':
                await loadDashboardPage();
                break;
            case 'calls':
                await loadCallsPage();
                break;
            case 'history':
                await loadHistoryPage();
                break;
            case 'extensions':
                await loadExtensionsPage();
                break;
            case 'trunks':
                await loadTrunksPage();
                break;
            case 'settings':
                await loadSettingsPage();
                break;
            case 'api-docs':
                await loadAPIDocsPage();
                break;
            default:
                contentDiv.innerHTML = '<h1>Page not found</h1>';
        }
    } catch (error) {
        console.error('Error loading page:', error);
        contentDiv.innerHTML = `<div class="error">Error loading page: ${error.message}</div>`;
    }
}

// Update active calls count
async function updateActiveCallsCount() {
    try {
        const calls = await API.getActiveCalls();
        const badge = document.getElementById('active-calls-count');
        if (badge) {
            badge.textContent = calls.length;
            badge.style.display = calls.length > 0 ? 'flex' : 'none';
        }
    } catch (error) {
        console.error('Failed to update calls count:', error);
    }
}

// Show notification
function showNotification(title, message, type = 'info') {
    const container = document.getElementById('notifications');
    
    const notification = document.createElement('div');
    notification.className = `notification glass-panel ${type} slide-in`;
    notification.innerHTML = `
        <div class="notification-header">
            <strong>${title}</strong>
            <button class="btn-icon" onclick="this.parentElement.parentElement.remove()">
                <i class="fap fa-times"></i>
            </button>
        </div>
        <div class="notification-body">${message}</div>
    `;
    
    container.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        notification.remove();
    }, 5000);
}

// Show modal
function showModal(title, content, actions = []) {
    const container = document.getElementById('modal-container');
    
    const modal = document.createElement('div');
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal glass-panel">
            <div class="modal-header">
                <h2 class="modal-title">${title}</h2>
                <button class="btn-icon" onclick="closeModal()">
                    <i class="fap fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                ${content}
            </div>
            <div class="modal-footer">
                ${actions.map(action => `
                    <button class="btn ${action.class || ''}" onclick="${action.onclick}">
                        ${action.icon ? `<i class="fap fa-${action.icon}"></i>` : ''}
                        <span>${action.label}</span>
                    </button>
                `).join('')}
            </div>
        </div>
    `;
    
    container.appendChild(modal);
}

// Close modal
function closeModal() {
    const container = document.getElementById('modal-container');
    container.innerHTML = '';
}

// Format phone number
function formatPhoneNumber(number) {
    if (!number) return '';
    
    // Remove non-numeric characters
    const cleaned = number.replace(/\D/g, '');
    
    // Format based on length
    if (cleaned.length === 10) {
        return `(${cleaned.slice(0, 3)}) ${cleaned.slice(3, 6)}-${cleaned.slice(6)}`;
    } else if (cleaned.length === 11 && cleaned[0] === '1') {
        return `+1 (${cleaned.slice(1, 4)}) ${cleaned.slice(4, 7)}-${cleaned.slice(7)}`;
    }
    
    return number;
}

// Format duration
function formatDuration(seconds) {
    if (!seconds) return '0:00';
    
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    
    if (hours > 0) {
        return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    }
    
    return `${minutes}:${secs.toString().padStart(2, '0')}`;
}

// Format date/time
function formatDateTime(dateString) {
    if (!dateString) return '';
    
    const date = new Date(dateString);
    const now = new Date();
    const diffMs = now - date;
    const diffMins = Math.floor(diffMs / 60000);
    
    // If less than 1 hour ago, show relative time
    if (diffMins < 60) {
        if (diffMins < 1) return 'just now';
        return `${diffMins} min${diffMins !== 1 ? 's' : ''} ago`;
    }
    
    // If today, show time
    if (date.toDateString() === now.toDateString()) {
        return date.toLocaleTimeString('en-US', { 
            hour: 'numeric', 
            minute: '2-digit' 
        });
    }
    
    // Otherwise show date and time
    return date.toLocaleDateString('en-US', {
        month: 'short',
        day: 'numeric',
        hour: 'numeric',
        minute: '2-digit'
    });
}

// Debounce function
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Export utilities
window.App = App;
window.showNotification = showNotification;
window.showModal = showModal;
window.closeModal = closeModal;
window.formatPhoneNumber = formatPhoneNumber;
window.formatDuration = formatDuration;
window.formatDateTime = formatDateTime;
window.debounce = debounce;