package config

import (
	"fmt"
	"os"

	"github.com/spf13/viper"
)

// Config represents the application configuration
type Config struct {
	Version string `yaml:"version"`
	
	Server struct {
		Host       string `yaml:"host"`
		Port       int    `yaml:"port"`
		TLSEnabled bool   `yaml:"tls_enabled"`
		TLSCert    string `yaml:"tls_cert"`
		TLSKey     string `yaml:"tls_key"`
	} `yaml:"server"`
	
	Database struct {
		Path string `yaml:"path"`
	} `yaml:"database"`
	
	Asterisk struct {
		AMIHost     string `yaml:"ami_host"`
		AMIPort     int    `yaml:"ami_port"`
		AMIUsername string `yaml:"ami_username"`
		AMIPassword string `yaml:"ami_password"`
	} `yaml:"asterisk"`
	
	API struct {
		Key         string   `yaml:"key"`
		CORSOrigins []string `yaml:"cors_origins"`
	} `yaml:"api"`
	
	Logging struct {
		Level      string `yaml:"level"`
		Path       string `yaml:"path"`
		MaxSize    int    `yaml:"max_size"`
		MaxBackups int    `yaml:"max_backups"`
	} `yaml:"logging"`
	
	Admin struct {
		Username string `yaml:"username"`
		Password string `yaml:"password"`
	} `yaml:"admin"`
}

// Load reads the configuration from file
func Load(configFile string) (*Config, error) {
	viper.SetConfigFile(configFile)
	viper.SetConfigType("yaml")
	
	// Set defaults
	viper.SetDefault("server.host", "0.0.0.0")
	viper.SetDefault("server.port", 8080)
	viper.SetDefault("server.tls_enabled", false)
	viper.SetDefault("database.path", "/var/lib/airwavepbx/airwavepbx.db")
	viper.SetDefault("asterisk.ami_host", "127.0.0.1")
	viper.SetDefault("asterisk.ami_port", 5038)
	viper.SetDefault("asterisk.ami_username", "airwave")
	viper.SetDefault("logging.level", "info")
	viper.SetDefault("logging.max_size", 100)
	viper.SetDefault("logging.max_backups", 7)
	
	// Read config file
	if err := viper.ReadInConfig(); err != nil {
		if _, ok := err.(viper.ConfigFileNotFoundError); ok {
			return nil, fmt.Errorf("configuration file not found: %s", configFile)
		}
		return nil, fmt.Errorf("failed to read configuration: %w", err)
	}
	
	var config Config
	if err := viper.Unmarshal(&config); err != nil {
		return nil, fmt.Errorf("failed to parse configuration: %w", err)
	}
	
	// Validate required fields
	if config.Admin.Username == "" || config.Admin.Password == "" {
		return nil, fmt.Errorf("admin credentials are required")
	}
	
	if config.API.Key == "" {
		return nil, fmt.Errorf("API key is required")
	}
	
	if config.Asterisk.AMIPassword == "" {
		return nil, fmt.Errorf("Asterisk AMI password is required")
	}
	
	// Expand environment variables
	config.Server.TLSCert = os.ExpandEnv(config.Server.TLSCert)
	config.Server.TLSKey = os.ExpandEnv(config.Server.TLSKey)
	config.Database.Path = os.ExpandEnv(config.Database.Path)
	config.Logging.Path = os.ExpandEnv(config.Logging.Path)
	
	return &config, nil
}