// AirwavePBX WebSocket Client

class AirwaveWebSocket {
    constructor() {
        this.ws = null;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 5;
        this.reconnectDelay = 1000;
        this.listeners = new Map();
        this.isConnected = false;
        this.shouldReconnect = true;
    }
    
    // Connect to WebSocket
    connect() {
        const token = localStorage.getItem('airwave_token');
        if (!token) {
            console.error('No authentication token available');
            return;
        }
        
        const protocol = window.location.protocol === 'https:' ? 'wss:' : 'ws:';
        const url = `${protocol}//${window.location.host}/ws?token=${token}`;
        
        try {
            this.ws = new WebSocket(url);
            this.setupEventHandlers();
        } catch (error) {
            console.error('WebSocket connection error:', error);
            this.handleReconnect();
        }
    }
    
    // Setup WebSocket event handlers
    setupEventHandlers() {
        this.ws.onopen = () => {
            console.log('WebSocket connected');
            this.isConnected = true;
            this.reconnectAttempts = 0;
            this.emit('connected');
            
            // Send ping every 30 seconds
            this.pingInterval = setInterval(() => {
                if (this.isConnected) {
                    this.send('ping', {});
                }
            }, 30000);
        };
        
        this.ws.onmessage = (event) => {
            try {
                const data = JSON.parse(event.data);
                this.handleMessage(data);
            } catch (error) {
                console.error('Failed to parse WebSocket message:', error);
            }
        };
        
        this.ws.onerror = (error) => {
            console.error('WebSocket error:', error);
            this.emit('error', error);
        };
        
        this.ws.onclose = () => {
            console.log('WebSocket disconnected');
            this.isConnected = false;
            this.emit('disconnected');
            
            if (this.pingInterval) {
                clearInterval(this.pingInterval);
                this.pingInterval = null;
            }
            
            if (this.shouldReconnect) {
                this.handleReconnect();
            }
        };
    }
    
    // Handle incoming messages
    handleMessage(data) {
        const { type, data: payload } = data;
        
        // Handle pong response
        if (type === 'pong') {
            return;
        }
        
        // Emit to listeners
        this.emit(type, payload);
        
        // Also emit a generic message event
        this.emit('message', data);
    }
    
    // Send message
    send(type, data) {
        if (!this.isConnected || !this.ws) {
            console.warn('WebSocket not connected');
            return;
        }
        
        const message = JSON.stringify({ type, data });
        this.ws.send(message);
    }
    
    // Event emitter methods
    on(event, callback) {
        if (!this.listeners.has(event)) {
            this.listeners.set(event, []);
        }
        this.listeners.get(event).push(callback);
    }
    
    off(event, callback) {
        if (!this.listeners.has(event)) return;
        
        const callbacks = this.listeners.get(event);
        const index = callbacks.indexOf(callback);
        if (index > -1) {
            callbacks.splice(index, 1);
        }
    }
    
    emit(event, data) {
        if (!this.listeners.has(event)) return;
        
        const callbacks = this.listeners.get(event);
        callbacks.forEach(callback => {
            try {
                callback(data);
            } catch (error) {
                console.error(`Error in event listener for ${event}:`, error);
            }
        });
    }
    
    // Reconnection logic
    handleReconnect() {
        if (this.reconnectAttempts >= this.maxReconnectAttempts) {
            console.error('Max reconnection attempts reached');
            this.emit('reconnect_failed');
            return;
        }
        
        this.reconnectAttempts++;
        const delay = this.reconnectDelay * Math.pow(2, this.reconnectAttempts - 1);
        
        console.log(`Reconnecting in ${delay}ms... (attempt ${this.reconnectAttempts})`);
        
        setTimeout(() => {
            if (this.shouldReconnect) {
                this.connect();
            }
        }, delay);
    }
    
    // Disconnect
    disconnect() {
        this.shouldReconnect = false;
        if (this.ws) {
            this.ws.close();
            this.ws = null;
        }
    }
    
    // Reconnect
    reconnect() {
        this.shouldReconnect = true;
        this.reconnectAttempts = 0;
        this.disconnect();
        this.connect();
    }
}

// Create global WebSocket instance
window.WS = new AirwaveWebSocket();

// Connect when authenticated
if (localStorage.getItem('airwave_token')) {
    WS.connect();
}

// Listen for call events
WS.on('call_ringing', (data) => {
    console.log('Incoming call:', data);
    showNotification('Incoming Call', `${data.caller_id_name || data.caller_id_number}`, 'info');
    
    // Update active calls count
    const badge = document.getElementById('active-calls-count');
    if (badge) {
        API.getActiveCalls().then(calls => {
            badge.textContent = calls.length;
        });
    }
});

WS.on('call_answered', (data) => {
    console.log('Call answered:', data);
});

WS.on('call_ended', (data) => {
    console.log('Call ended:', data);
    
    // Update active calls count
    const badge = document.getElementById('active-calls-count');
    if (badge) {
        API.getActiveCalls().then(calls => {
            badge.textContent = calls.length;
        });
    }
});